{

Minor changes have been done to have different names for the 32 and 64 bit DLLs to
avoid "DLL hell".


Small changes for Delphi 7, 2007 compatibility where String is AnsiString


https://github.com/ahausladen/PdfiumLib

https://github.com/PDFium/PDFium/blob/master/LICENSE
}


{$A8,B-,E-,F-,G+,H+,I+,J-,K-,M-,N-,P+,Q-,R-,S-,T-,U-,V+,X+,Z1}


unit PdfiumCore;

{$I Envision.Inc}

interface

uses

    {$ifdef __DelphiD13AndAbove}Winapi.Windows,{$else}Windows,{$endif}

  Types, SysUtils, Classes, Contnrs,

  PdfiumAPI;

type
  EPdfException = class(Exception);
  EPdfUnsupportedFeatureException = class(EPdfException);
  EPdfArgumentOutOfRange = class(EPdfException);

  TPdfDocument = class;
  TPdfPage = class;

  TPdfPoint = {$if CompilerVersion >= 17} record {$else} object {$ifend}
    X, Y: Double;
    class function Empty: TPdfPoint; {$if CompilerVersion >= 17} static; {$ifend}
    procedure Offset(XOffset, YOffset: Double);
  end;

  TPdfRect = record

     Width: Double;
     Height: Double;

    {$WARNINGS OFF}
    case Coords:Integer of
      0: (Left, Top, Right, Bottom: Double);
      1: (TopLeft: TPdfPoint; BottomRight: TPdfPoint);
    {$WARNINGS ON}

  end;

  TPdfDocumentCustomReadProc = function(Param: Pointer; Position: LongWord; Buffer: PByte; Size: LongWord): Boolean;

  TPdfPageRenderOptionType = (
    proAnnotations,            // Set if annotations are to be rendered.
    proLCDOptimized,           // Set if using text rendering optimized for LCD display.
    proNoNativeText,           // Don't use the native text output available on some platforms
    proGrayscale,              // Grayscale output.
    proNoCatch,                // Set if you don't want to catch exception.
    proLimitedImageCacheSize,  // Limit image cache size.
    proForceHalftone,          // Always use halftone for image stretching.
    proPrinting,               // Render for printing.
    proReverseByteOrder        // Set whether render in a reverse Byte order, this flag only enable when render to a bitmap.
  );
  TPdfPageRenderOptions = set of TPdfPageRenderOptionType;

  TPdfPageRotation = (
    prNormal             = 0,
    pr90Clockwise        = 1,
    pr180                = 2,
    pr90CounterClockwide = 3
  );

  TPdfDocumentSaveOption = (
    dsoIncremental    = 1,
    dsoNoIncremental  = 2,
    dsoRemoveSecurity = 3
  );

  TPdfDocumentLoadOption = (
    dloNormal,   // load the whole file into memory
    dloMMF,      // load the file by using a memory mapped file (file stays open)
    dloOnDemand  // load the file using the custom load function (file stays open)
  );

  TPdfDocumentPageMode = (
    dpmUnknown        = -1, // Unknown value
    dpmUseNone        = 0,  // Neither document outline nor thumbnail images visible
    dpmUseOutlines    = 1,  // Document outline visible
    dpmUseThumbs      = 2,  // Thumbnial images visible
    dpmFullScreen     = 3,  // Full-screen mode, with no menu bar, window controls, or any other window visible
    dpmUseOC          = 4,  // Optional content group panel visible
    dpmUseAttachments = 5   // Attachments panel visible
  );

  TPdfBitmapFormat = (
    bfGrays = FPDFBitmap_Gray, // Gray scale bitmap, one byte per pixel.
    bfBGR   = FPDFBitmap_BGR,  // 3 bytes per pixel, byte order: blue, green, red.
    bfBGRx  = FPDFBitmap_BGRx, // 4 bytes per pixel, byte order: blue, green, red, unused.
    bfBGRA  = FPDFBitmap_BGRA  // 4 bytes per pixel, byte order: blue, green, red, alpha.
  );

  _TPdfBitmapHideCtor = class(TObject)
  private
    procedure Create;
  end;

  TPdfBitmap = class(_TPdfBitmapHideCtor)
  private
    FBitmap: FPDF_BITMAP;
    FOwnsBitmap: Boolean;
    FWidth: Integer;
    FHeight: Integer;
    FBytesPerScanLine: Integer;
  public
    constructor Create(ABitmap: FPDF_BITMAP; AOwnsBitmap: Boolean = False); overload;
    constructor Create(AWidth, AHeight: Integer; AAlpha: Boolean); overload;
    constructor Create(AWidth, AHeight: Integer; AFormat: TPdfBitmapFormat); overload;
    constructor Create(AWidth, AHeight: Integer; AFormat: TPdfBitmapFormat; ABuffer: Pointer; ABytesPerScanline: Integer); overload;
    destructor Destroy; override;

    procedure FillRect(ALeft, ATop, AWidth, AHeight: Integer; AColor: FPDF_DWORD);
    function GetBuffer: Pointer;

    property Width: Integer read FWidth;
    property Height: Integer read FHeight;
    property BytesPerScanline: Integer read FBytesPerScanLine;
    property Bitmap: FPDF_BITMAP read FBitmap;
  end;

  TPdfPage = class(TObject)
  private
    FDocument: TPdfDocument;
    FPage: FPDF_PAGE;
    FWidth: Double;
    FHeight: Double;
    FTransparency: Boolean;
    FRotation: TPdfPageRotation;
    FTextHandle: FPDF_TEXTPAGE;
    FSearchHandle: FPDF_SCHHANDLE;
    FLinkHandle: FPDF_PAGELINK;
    constructor Create(ADocument: TPdfDocument; APage: FPDF_PAGE);
    procedure UpdateMetrics;
    procedure Open;
    procedure SetRotation(const Value: TPdfPageRotation);
    function BeginText: Boolean;
    function BeginWebLinks: Boolean;
    class function GetDrawFlags(const Options: TPdfPageRenderOptions): Integer; {$if CompilerVersion >= 17} static; {$ifend}
  public
    destructor Destroy; override;
    procedure Close;

    procedure Draw(DC: HDC; X, Y, Width, Height: Integer; Rotate: TPdfPageRotation = prNormal;
      const Options: TPdfPageRenderOptions = []);
    procedure DrawToPdfBitmap(APdfBitmap: TPdfBitmap; X, Y, Width, Height: Integer; Rotate: TPdfPageRotation = prNormal;
      const Options: TPdfPageRenderOptions = []);

    function DeviceToPage(X, Y, Width, Height: Integer; DeviceX, DeviceY: Integer; Rotate: TPdfPageRotation = prNormal): TPdfPoint; overload;
    function PageToDevice(X, Y, Width, Height: Integer; PageX, PageY: Double; Rotate: TPdfPageRotation = prNormal): TPoint; overload;
    function DeviceToPage(X, Y, Width, Height: Integer; const R: TRect; Rotate: TPdfPageRotation = prNormal): TPdfRect; overload;
    function PageToDevice(X, Y, Width, Height: Integer; const R: TPdfRect; Rotate: TPdfPageRotation = prNormal): TRect; overload;

    procedure ApplyChanges;

    function BeginFind(const SearchString: WideString; MatchCase, MatchWholeWord: Boolean; FromEnd: Boolean): Boolean;
    function FindNext(var CharIndex, Count: Integer): Boolean;
    function FindPrev(var CharIndex, Count: Integer): Boolean;
    procedure EndFind;

    function GetCharCount: Integer;
    function ReadChar(CharIndex: Integer): WideChar;
    function GetCharFontSize(CharIndex: Integer): Double;
    function GetCharBox(CharIndex: Integer): TPdfRect;
    function GetCharIndexAt(PageX, PageY, ToleranceX, ToleranceY: Double): Integer;
    function ReadText(CharIndex, Count: Integer): Widestring;
    function GetTextAt(const R: TPdfRect): Widestring; overload;
    function GetTextAt(Left, Top, Right, Bottom: Double): Widestring; overload;

    function GetTextRectCount(CharIndex, Count: Integer): Integer;
    function GetTextRect(RectIndex: Integer): TPdfRect;

    function GetWebLinkCount: Integer;
    function GetWebLinkURL(LinkIndex: Integer): Widestring;
    function GetWebLinkRectCount(LinkIndex: Integer): Integer;
    function GetWebLinkRect(LinkIndex, RectIndex: Integer): TPdfRect;

    property Width: Double read FWidth;
    property Height: Double read FHeight;
    property Transparency: Boolean read FTransparency;
    property Rotation: TPdfPageRotation read FRotation write SetRotation;
  end;

  TBytes = array of Byte;

    PCustomLoadDataRec = ^TCustomLoadDataRec;
    TCustomLoadDataRec = record
      Param: Pointer;
      GetBlock: TPdfDocumentCustomReadProc;
      FileAccess: TFPDFFileAccess;
    end;

  TPdfDocument = class(TObject)
  private
    FDocument: FPDF_DOCUMENT;

    FLastPage : Integer;

    FPages: TObjectList;
    FFileName: string;
    FFileHandle: THandle;
    FFileMapping: THandle;
    FBuffer: PByte;
    FBytes: TBytes;
    FClosing: Boolean;
    FCustomLoadData: PCustomLoadDataRec;

    procedure InternLoadFromMem(Buffer: PByte; Size: Integer; const APassword: AnsiString);
    procedure InternLoadFromCustom(ReadFunc: TPdfDocumentCustomReadProc; ASize: LongWord; AParam: Pointer; const APassword: AnsiString);
    function GetPage(Index: Integer): TPdfPage;
    function GetPageCount: Integer;
    procedure ExtractPage(APage: TPdfPage);
    function ReloadPage(APage: TPdfPage): FPDF_PAGE;
    function GetActive: Boolean;
    procedure CheckActive;
    function GetFileVersion: Integer;
    function GetPageSize(Index: Integer): TPdfPoint;
  protected
    property Document: FPDF_DOCUMENT read FDocument;
  public
    constructor Create;
    destructor Destroy; override;

    procedure LoadFromCustom(ReadFunc: TPdfDocumentCustomReadProc; ASize: LongWord; AParam: Pointer; const APassword: AnsiString = '');
    procedure LoadFromActiveStream(Stream: TStream; const APassword: AnsiString = ''); // Stream must not be released until the document is closed
    procedure LoadFromActiveBuffer(Buffer: Pointer; Size: Integer; const APassword: AnsiString = ''); // Buffer must not be released until the document is closed
    procedure LoadFromBytes(const ABytes: TBytes; const APassword: AnsiString = ''); overload;
    procedure LoadFromBytes(const ABytes: TBytes; AIndex: Integer; ACount: Integer; const APassword: AnsiString = ''); overload;
    procedure LoadFromStream(AStream: TStream; const APassword: AnsiString = '');
    procedure LoadFromFile(const AFileName: string; const APassword: AnsiString = ''; ALoadOptions: TPdfDocumentLoadOption = dloMMF);
    procedure Close;

    procedure SaveToFile(const AFileName: string; Option: TPdfDocumentSaveOption = dsoRemoveSecurity; FileVersion: Integer = -1);
    procedure SaveToStream(Stream: TStream; Option: TPdfDocumentSaveOption = dsoRemoveSecurity; FileVersion: Integer = -1);

    function NewDocument: Boolean;
    procedure DeletePage(Index: Integer);
    function NewPage(Width, Height: Double; Index: Integer = -1): TPdfPage;

    function GetMetaText(const TagName: string): WideString;

    property FileName: string read FFileName;
    property PageCount: Integer read GetPageCount;
    property Pages[Index: Integer]: TPdfPage read GetPage;
    property PageSizes[Index: Integer]: TPdfPoint read GetPageSize;

    property Active: Boolean read GetActive;
    property FileVersion: Integer read GetFileVersion;
  end;

var
  PDFiumDllDir: string = '';

implementation



resourcestring
  RsUnsupportedFeature = 'Function %s not supported';
  RsArgumentsOutOfRange = 'Functions argument "%s" out of range';
  RsDocumentNotActive = 'PDF document is not open';
  RsFileTooLarge = 'PDF file "%s" is too large';

  RsPdfErrorSuccess  = 'No error';
  RsPdfErrorUnknown  = 'Unknown error';
  RsPdfErrorFile     = 'File not found or can''t be opened';
  RsPdfErrorFormat   = 'File is not a PDF document or is corrupted';
  RsPdfErrorPassword = 'Password required, invalid password';
  RsPdfErrorSecurity = 'Security schema is not support';
  RsPdfErrorPage     = 'Page does not exist or data error';

procedure UnsupportedHandler(pThis: PUNSUPPORT_INFO; nType: Integer); cdecl;
var
  Typ: string;
begin
  case nType of
    FPDF_UNSP_DOC_XFAFORM:
      Typ := 'XFA';

    FPDF_UNSP_DOC_PORTABLECOLLECTION:
      Typ := 'Portfolios_Packages';

    FPDF_UNSP_DOC_ATTACHMENT,
    FPDF_UNSP_ANNOT_ATTACHMENT:
      Typ := 'Attachment';

    FPDF_UNSP_DOC_SECURITY:
      Typ := 'Rights_Management';

    FPDF_UNSP_DOC_SHAREDREVIEW:
      Typ := 'Shared_Review';

    FPDF_UNSP_DOC_SHAREDFORM_ACROBAT,
    FPDF_UNSP_DOC_SHAREDFORM_FILESYSTEM,
    FPDF_UNSP_DOC_SHAREDFORM_EMAIL:
      Typ := 'Shared_Form';

    FPDF_UNSP_ANNOT_3DANNOT:
      Typ := '3D';

    FPDF_UNSP_ANNOT_MOVIE:
      Typ := 'Movie';

    FPDF_UNSP_ANNOT_SOUND:
      Typ := 'Sound';

    FPDF_UNSP_ANNOT_SCREEN_MEDIA,
    FPDF_UNSP_ANNOT_SCREEN_RICHMEDIA:
      Typ := 'Screen';

    FPDF_UNSP_ANNOT_SIG:
      Typ := 'Digital_Signature';

  else
    Typ := 'Unknown';
  end;
  raise EPdfUnsupportedFeatureException.CreateResFmt(@RsUnsupportedFeature, [Typ]);
end;

var
  PDFiumInitCritSect: TRTLCriticalSection;
  UnsupportInfo: TUnsupportInfo = (
    version: 1;
    FSDK_UnSupport_Handler: UnsupportedHandler;
  );

procedure InitLib;
{$J+}
const
  Initialized: Integer = 0;
{$J-}
begin
  if Initialized = 0 then
  begin
    EnterCriticalSection(PDFiumInitCritSect);
    try
      if Initialized = 0 then
      begin
        InitPDFium(PDFiumDllDir);
        PdfSetUnSpObjProcessHandler(@UnsupportInfo);
        Initialized := 1;
      end;
    finally
      LeaveCriticalSection(PDFiumInitCritSect);
    end;
  end;
end;

procedure RaiseLastPdfError;
begin
  case PdfGetLastError of
    FPDF_ERR_SUCCESS:
      raise EPdfException.CreateRes(@RsPdfErrorSuccess);
    FPDF_ERR_FILE:
      raise EPdfException.CreateRes(@RsPdfErrorFile);
    FPDF_ERR_FORMAT:
      raise EPdfException.CreateRes(@RsPdfErrorFormat);
    FPDF_ERR_PASSWORD:
      raise EPdfException.CreateRes(@RsPdfErrorPassword);
    FPDF_ERR_SECURITY:
      raise EPdfException.CreateRes(@RsPdfErrorSecurity);
    FPDF_ERR_PAGE:
      raise EPdfException.CreateRes(@RsPdfErrorPage);
  else
    raise EPdfException.CreateRes(@RsPdfErrorUnknown);
  end;
end;

{ TPdfRect }

procedure EmptyPdfRect( var PdfRect : TPdfRect );
begin
    PdfRect.Left := 0;
    PdfRect.Top := 0;
    PdfRect.Right := 0;
    PdfRect.Bottom := 0;
end;


{ TPdfDocument }

constructor TPdfDocument.Create;
begin
  inherited Create;
  InitLib;
  FFileHandle := INVALID_HANDLE_VALUE;
  FPages := TObjectList.Create;
  FLastPage := -1;
end;

destructor TPdfDocument.Destroy;
begin
  if FPages <> nil then
      Close;
  FPages.Free;
  inherited Destroy;
end;

procedure TPdfDocument.Close;
begin
  FClosing := True;
  try
    FPages.Clear;

    if FDocument <> nil then
    begin
      PdfCloseDocument(FDocument);
      FDocument := nil;
    end;

    if FCustomLoadData <> nil then
    begin
      Dispose(FCustomLoadData);
      FCustomLoadData := nil;
    end;

    if FFileMapping <> 0 then
    begin
      if FBuffer <> nil then
      begin
        UnmapViewOfFile(FBuffer);
        FBuffer := nil;
      end;
      CloseHandle(FFileMapping);
      FFileMapping := 0;
    end
    else if FBuffer <> nil then
    begin
      FreeMem(FBuffer);
      FBuffer := nil;
    end;
    FBytes := nil;

    if FFileHandle <> INVALID_HANDLE_VALUE then
    begin
      CloseHandle(FFileHandle);
      FFileHandle := INVALID_HANDLE_VALUE;
    end;

    FFileName := '';
  finally
    FClosing := False;
  end;
end;

function ReadFromActiveFile(Param: Pointer; Position: LongWord; Buffer: PByte; Size: LongWord): Boolean;
var
  NumRead: DWORD;
begin
  if Buffer <> nil then
  begin
    SetFilePointer(THandle(Param), Position, nil, FILE_BEGIN);
    Result := ReadFile(THandle(Param), Buffer^, Size, NumRead, nil) and (NumRead = Size);
  end
  else
    Result := Size = 0;
end;

procedure TPdfDocument.LoadFromFile(const AFileName: string; const APassword: AnsiString; ALoadOptions: TPdfDocumentLoadOption);
var
  Size, Offset: NativeInt;
  NumRead: DWORD;
  SizeHigh: DWORD;
begin
  Close;

  FFileHandle := CreateFile(PChar(AFileName), GENERIC_READ, FILE_SHARE_READ, nil, OPEN_EXISTING, FILE_ATTRIBUTE_NORMAL, 0);
  if FFileHandle = INVALID_HANDLE_VALUE then
    RaiseLastOSError;
  try
    Size := GetFileSize(FFileHandle, @SizeHigh);
    if SizeHigh <> 0 then
      raise EPdfException.CreateResFmt(@RsFileTooLarge, [ExtractFileName(AFileName)]);

    case ALoadOptions of
      dloNormal:
        begin
          if Size > 0 then
          begin
            GetMem(FBuffer, Size);
            Offset := 0;
            while Offset < Size do
            begin
              if ((Size - Offset) and not $FFFFFFFF) <> 0 then
                NumRead := $40000000
              else
                NumRead := Size - Offset;

              if not ReadFile(FFileHandle, TBytes(FBuffer^)[Offset], NumRead, NumRead, nil) then
                RaiseLastOSError;
              Inc(Offset, NumRead);
            end;

            InternLoadFromMem(FBuffer, Size, APassword);
          end;
        end;

      dloMMF:
        begin
          FFileMapping := CreateFileMapping(FFileHandle, nil, PAGE_READONLY, 0, Size, nil);
          if FFileMapping = 0 then
            RaiseLastOSError;
          FBuffer := MapViewOfFile(FFileMapping, FILE_MAP_READ, 0, 0, Size);
          if FBuffer = nil then
            RaiseLastOSError;

          InternLoadFromMem(FBuffer, Size, APassword);
        end;

      dloOnDemand:                      
        InternLoadFromCustom(ReadFromActiveFile, Size, Pointer(FFileHandle), APassword);
    end;
  except
    Close;
    raise;
  end;
  FFileName := AFileName;
end;

procedure TPdfDocument.LoadFromStream(AStream: TStream; const APassword: AnsiString);
var
  Size: NativeInt;
begin
  Close;
  Size := AStream.Size;
  if Size > 0 then
  begin
    GetMem(FBuffer, Size);
    try
      AStream.ReadBuffer(FBuffer^, Size);
      InternLoadFromMem(FBuffer, Size, APassword);
    except
      Close;
      raise;
    end;
  end;
end;

procedure TPdfDocument.LoadFromActiveBuffer(Buffer: Pointer; Size: Integer; const APassword: AnsiString);
begin
  Close;
  InternLoadFromMem(Buffer, Size, APassword);
end;

procedure TPdfDocument.LoadFromBytes(const ABytes: TBytes; const APassword: AnsiString);
begin
  LoadFromBytes(ABytes, 0, Length(ABytes), APassword);
end;

procedure TPdfDocument.LoadFromBytes(const ABytes: TBytes; AIndex, ACount: Integer;
  const APassword: AnsiString);
var
  Len: Integer;
begin
  Close;

  Len := Length(ABytes);
  if AIndex >= Len then
    raise EPdfArgumentOutOfRange.CreateResFmt(@RsArgumentsOutOfRange, ['Index']);
  if AIndex + ACount > Len then
    raise EPdfArgumentOutOfRange.CreateResFmt(@RsArgumentsOutOfRange, ['Count']);

  FBytes := ABytes; // keep alive after return
  InternLoadFromMem(@ABytes[AIndex], ACount, APassword);
end;

function ReadFromActiveStream(Param: Pointer; Position: LongWord; Buffer: PByte; Size: LongWord): Boolean;
begin
  if Buffer <> nil then
  begin
    {$WARNINGS OFF}
    TStream(Param).Seek(Position, soBeginning);
    Result := TStream(Param).Read(Buffer^, Size) = Integer(Size);
    {$WARNINGS ON}
  end
  else
    Result := Size = 0;
end;

procedure TPdfDocument.LoadFromActiveStream(Stream: TStream; const APassword: AnsiString);
begin
  if Stream = nil then
    Close
  else
    {$Warnings OFF}
    LoadFromCustom(ReadFromActiveStream, Stream.Size, Stream, APassword);
    {$Warnings ON}
end;

procedure TPdfDocument.LoadFromCustom(ReadFunc: TPdfDocumentCustomReadProc; ASize: LongWord;
  AParam: Pointer; const APassword: AnsiString);
begin
  Close;
  InternLoadFromCustom(ReadFunc, ASize, AParam, APassword);
end;

function GetLoadFromCustomBlock(Param: Pointer; Position: LongWord; Buffer: PByte; Size: LongWord): Integer; cdecl;
begin
  {$Warnings OFF}
  Result := Ord(TPdfDocument(param).FCustomLoadData.GetBlock(TPdfDocument(param).FCustomLoadData.Param, Position, Buffer, Size));
  {$Warnings ON}
end;

procedure TPdfDocument.InternLoadFromCustom(ReadFunc: TPdfDocumentCustomReadProc; ASize: LongWord;
  AParam: Pointer; const APassword: AnsiString);
begin
  if Assigned(ReadFunc) then
  begin
    New(FCustomLoadData);
    FCustomLoadData.Param := AParam;
    FCustomLoadData.GetBlock := ReadFunc;
    FCustomLoadData.FileAccess.m_FileLen := ASize;
    FCustomLoadData.FileAccess.m_GetBlock := GetLoadFromCustomBlock;
{$WARNINGS OFF}
    FCustomLoadData.FileAccess.m_Param := Self;
{$WARNINGS ON}

    FDocument := PdfLoadCustomDocument(@FCustomLoadData.FileAccess, PAnsiChar(APassword));
    if FDocument = nil then
      RaiseLastPdfError;

    FPages.Count := PdfGetPageCount(FDocument);
  end;
end;

procedure TPdfDocument.InternLoadFromMem(Buffer: PByte; Size: Integer; const APassword: AnsiString);
begin
  if Size > 0 then
  begin
    FDocument := PdfLoadMemDocument(Buffer, Size, PAnsiChar(Pointer(APassword)));
    if FDocument = nil then
      RaiseLastPdfError;

    FPages.Count := PdfGetPageCount(FDocument);
  end;
end;

function TPdfDocument.GetPage(Index: Integer): TPdfPage;
var
  LPage  : FPDF_PAGE;
begin

  if (Index < 0) or (Index >= FPages.Count) then
  begin
      Result := nil;
      Exit;
  end;

  if (FLastPage <> -1) and (FLastPage <> Index) then
  begin
      if FPages[FLastPage] <> nil then
      begin
          { this will free the page }
          FPages[FLastPage] := nil;
      end;
  end;

  Result := TPdfPage(FPages[Index]);
  if Result = nil then
  begin
    LPage := PdfLoadPage(FDocument, Index);
    if LPage = nil then
      RaiseLastPdfError;
    Result := TPdfPage.Create(Self, LPage);
    FPages[Index] := Result;
  end;

  FLastPage := Index;
end;

function TPdfDocument.GetPageCount: Integer;
begin
  Result := FPages.Count;
end;

procedure TPdfDocument.ExtractPage(APage: TPdfPage);
begin
  if not FClosing then
    FPages.Extract(APage);
end;

function TPdfDocument.ReloadPage(APage: TPdfPage): FPDF_PAGE;
var
  Index: Integer;
begin
  CheckActive;
  Index := FPages.IndexOf(APage);
  Result := PdfLoadPage(FDocument, Index);
  if Result = nil then
    RaiseLastPdfError;
end;

function TPdfDocument.GetActive: Boolean;
begin
  Result := FDocument <> nil;
end;

procedure TPdfDocument.CheckActive;
begin
  if not Active then
    raise EPdfException.CreateRes(@RsDocumentNotActive);
end;

procedure TPdfDocument.SaveToFile(const AFileName: string; Option: TPdfDocumentSaveOption; FileVersion: Integer);
var
  Stream: TFileStream;
begin
  Stream := TFileStream.Create(AFileName, fmCreate or fmShareDenyWrite);
  try
    SaveToStream(Stream, Option, FileVersion);
  finally
    Stream.Free;
  end;
end;

type
  PFPDFFileWriteEx = ^TFPDFFileWriteEx;
  TFPDFFileWriteEx = record
    Inner: TFPDFFileWrite;
    Stream: TStream;
  end;

function WriteBlockToStream(pThis: PFPDF_FILEWRITE; pData: Pointer; size: LongWord): Integer; cdecl;
begin
  Result := Ord(LongWord(PFPDFFileWriteEx(pThis).Stream.Write(pData^, size)) = size);
end;

procedure TPdfDocument.SaveToStream(Stream: TStream; Option: TPdfDocumentSaveOption; FileVersion: Integer);
var
  FileWriteInfo: TFPDFFileWriteEx;
begin
  FileWriteInfo.Inner.version := 1;
  FileWriteInfo.Inner.WriteBlock := @WriteBlockToStream;
  FileWriteInfo.Stream := Stream;

  if FileVersion <> -1 then
    PdfSaveWithVersion(FDocument, @FileWriteInfo, Ord(Option), FileVersion)
  else
    PdfSaveAsCopy(FDocument, @FileWriteInfo, Ord(Option));
end;

(*
procedure TPdfDocument.SaveToBytes(var Bytes: TBytes; Option: TPdfDocumentSaveOption; FileVersion: Integer);
var
  Stream: TBytesStream;
begin
  Stream := TBytesStream.Create(nil);
  try
    SaveToStream(Stream, Option, FileVersion);
    Bytes := Stream.Bytes;
  finally
    Stream.Free;
  end;
end;
*)

function TPdfDocument.NewDocument: Boolean;
begin
  Close;
  FDocument := PdfCreateNewDocument;
  Result := FDocument <> nil;
end;

procedure TPdfDocument.DeletePage(Index: Integer);
begin
  CheckActive;
  FPages.Delete(Index);
  PdfPage_Delete(FDocument, Index);
end;

function TPdfDocument.NewPage(Width, Height: Double; Index: Integer): TPdfPage;
var
  LPage: FPDF_PAGE;
begin
  CheckActive;
  if Index < 0 then
    Index := FPages.Count;
  LPage := PdfPage_New(FDocument, Index, Width, Height);
  if LPage <> nil then
  begin
    Result := TPdfPage.Create(Self, LPage);
    FPages.Insert(Index, Result);
  end
  else
    Result := nil;
end;

function TPdfDocument.GetMetaText(const TagName: string): WideString;
var
  Len: Integer;
  A: AnsiString;
begin
  CheckActive;
  {$WARNINGS OFF}
  A := AnsiString(TagName);
  {$WARNINGS ON}

  Len := (PdfGetMetaText(FDocument, PAnsiChar(A), nil, 0) div SizeOf(WideChar)) - 1;
  if Len > 0 then
  begin
    SetLength(Result, Len);
    PdfGetMetaText(FDocument, PAnsiChar(A), PWideChar(Result), (Len + 1) * SizeOf(WideChar));
  end
  else
    Result := '';
end;

function TPdfDocument.GetFileVersion: Integer;
begin
  CheckActive;
  if PdfGetFileVersion(FDocument, Result) = 0 then
    Result := 0;
end;

function TPdfDocument.GetPageSize(Index: Integer): TPdfPoint;
begin
  CheckActive;
  if PdfGetPageSizeByIndex(FDocument, Index, Result.X, Result.Y) = 0 then
  begin
    Result.X := 0;
    Result.Y := 0;
  end;
end;


{ TPdfPage }

constructor TPdfPage.Create(ADocument: TPdfDocument; APage: FPDF_PAGE);
begin
  inherited Create;
  FDocument := ADocument;
  FPage := APage;
  UpdateMetrics;
end;

destructor TPdfPage.Destroy;
begin
  Close;
  FDocument.ExtractPage(Self);
  inherited Destroy;
end;

procedure TPdfPage.Close;
begin
  if FLinkHandle <> nil then
  begin
    PdfLink_CloseWebLinks(FLinkHandle);
    FLinkHandle := nil;
  end;
  if FSearchHandle <> nil then
  begin
    PdfText_FindClose(FSearchHandle);
    FSearchHandle := nil;
  end;
  if FTextHandle <> nil then
  begin
    PdfText_ClosePage(FTextHandle);
    FTextHandle := nil;
  end;
  if FPage <> nil then
  begin
    PdfClosePage(FPage);
    FPage := nil;
  end
end;

procedure TPdfPage.Open;
begin
  if FPage = nil then
  begin
    FPage := FDocument.ReloadPage(Self);
    UpdateMetrics;
  end;
end;

class function TPdfPage.GetDrawFlags(const Options: TPdfPageRenderOptions): Integer;
begin
  Result := 0;
  if proAnnotations in Options then
    Result := Result or FPDF_ANNOT;
  if proLCDOptimized in Options then
    Result := Result or FPDF_LCD_TEXT;
  if proNoNativeText in Options then
    Result := Result or FPDF_NO_NATIVETEXT;
  if proGrayscale in Options then
    Result := Result or FPDF_GRAYSCALE;
  if proNoCatch in Options then
    Result := Result or FPDF_NO_CATCH;
  if proLimitedImageCacheSize in Options then
    Result := Result or FPDF_RENDER_LIMITEDIMAGECACHE;
  if proForceHalftone in Options then
    Result := Result or FPDF_RENDER_FORCEHALFTONE;
  if proPrinting in Options then
    Result := Result or FPDF_PRINTING;
  if proReverseByteOrder in Options then
    Result := Result or FPDF_REVERSE_BYTE_ORDER;
end;

procedure TPdfPage.Draw(DC: HDC; X, Y, Width, Height: Integer; Rotate: TPdfPageRotation; const Options: TPdfPageRenderOptions);
begin
  Open;
  PdfRenderPage(DC, FPage, X, Y, Width, Height, Ord(Rotate), GetDrawFlags(Options));
end;

procedure TPdfPage.DrawToPdfBitmap(APdfBitmap: TPdfBitmap; X, Y, Width, Height: Integer;
  Rotate: TPdfPageRotation; const Options: TPdfPageRenderOptions);
begin
  Open;
  PdfRenderPageBitmap(APdfBitmap.FBitmap, FPage, X, Y, Width, Height, Ord(Rotate), GetDrawFlags(Options));
end;

procedure TPdfPage.UpdateMetrics;
begin
  FWidth := PdfGetPageWidth(FPage);
  FHeight := PdfGetPageHeight(FPage);
  FTransparency := PdfPage_HasTransparency(FPage) <> 0;
  FRotation := TPdfPageRotation(PdfPage_GetRotation(FPage));
end;

function TPdfPage.DeviceToPage(X, Y, Width, Height: Integer; DeviceX, DeviceY: Integer; Rotate: TPdfPageRotation): TPdfPoint;
begin
  Open;
  PdfDeviceToPage(FPage, X, Y, Width, Height, Ord(Rotate), DeviceX, DeviceY, Result.X, Result.Y);
end;

function TPdfPage.PageToDevice(X, Y, Width, Height: Integer; PageX, PageY: Double;
  Rotate: TPdfPageRotation): TPoint;
begin
  Open;
  PdfPageToDevice(FPage, X, Y, Width, Height, Ord(Rotate), PageX, PageY, Result.X, Result.Y);
end;

function TPdfPage.DeviceToPage(X, Y, Width, Height: Integer; const R: TRect; Rotate: TPdfPageRotation): TPdfRect;
begin
  Result.TopLeft := DeviceToPage(X, Y, Width, Height, R.Left, R.Top, Rotate);
  Result.BottomRight := DeviceToPage(X, Y, Width, Height, R.Right, R.Bottom, Rotate);
end;

function TPdfPage.PageToDevice(X, Y, Width, Height: Integer; const R: TPdfRect; Rotate: TPdfPageRotation): TRect;
begin
  Result.TopLeft := PageToDevice(X, Y, Width, Height, R.Left, R.Top, Rotate);
  Result.BottomRight := PageToDevice(X, Y, Width, Height, R.Right, R.Bottom, Rotate);
end;

procedure TPdfPage.SetRotation(const Value: TPdfPageRotation);
begin
  Open;
  PdfPage_SetRotation(FPage, Ord(Value));
  FRotation := TPdfPageRotation(PdfPage_GetRotation(FPage));
end;

procedure TPdfPage.ApplyChanges;
begin
  if FPage <> nil then
    PdfPage_GenerateContent(FPage);
end;

function TPdfPage.BeginText: Boolean;
begin
  if FTextHandle = nil then
  begin
    Open;
    FTextHandle := PdfText_LoadPage(FPage);
  end;
  Result := FTextHandle <> nil;
end;

function TPdfPage.BeginWebLinks: Boolean;
begin
  if (FLinkHandle = nil) and BeginText then
    FLinkHandle := PdfLink_LoadWebLinks(FTextHandle);
  Result := FLinkHandle <> nil;
end;

function TPdfPage.BeginFind(const SearchString: WideString; MatchCase, MatchWholeWord,
  FromEnd: Boolean): Boolean;
var
  Flags, StartIndex: Integer;
begin
  EndFind;
  if BeginText then
  begin
    Flags := 0;
    if MatchCase then
      Flags := Flags or FPDF_MATCHCASE;
    if MatchWholeWord then
      Flags := Flags or FPDF_MATCHWHOLEWORD;

    if FromEnd then
      StartIndex := -1
    else
      StartIndex := 0;

    FSearchHandle := PdfText_FindStart(FTextHandle, PWideChar(SearchString), Flags, StartIndex);
  end;
  Result := FSearchHandle <> nil;
end;

procedure TPdfPage.EndFind;
begin
  if FSearchHandle <> nil then
  begin
    PdfText_FindClose(FSearchHandle);
    FSearchHandle := nil;
  end;
end;

function TPdfPage.FindNext(var CharIndex, Count: Integer): Boolean;
begin
  CharIndex := 0;
  Count := 0;
  if FSearchHandle <> nil then
  begin
    Result := PdfText_FindNext(FSearchHandle) <> 0;
    if Result then
    begin
      CharIndex := PdfText_GetSchResultIndex(FSearchHandle);
      Count := PdfText_GetSchCount(FSearchHandle);
    end;
  end
  else
    Result := False;
end;

function TPdfPage.FindPrev(var CharIndex, Count: Integer): Boolean;
begin
  CharIndex := 0;
  Count := 0;
  if FSearchHandle <> nil then
  begin
    Result := PdfText_FindPrev(FSearchHandle) <> 0;
    if Result then
    begin
      CharIndex := PdfText_GetSchResultIndex(FSearchHandle);
      Count := PdfText_GetSchCount(FSearchHandle);
    end;
  end
  else
    Result := False;
end;

function TPdfPage.GetCharCount: Integer;
begin
  if BeginText then
    Result := PdfText_CountChars(FTextHandle)
  else
    Result := 0;
end;

function TPdfPage.ReadChar(CharIndex: Integer): WideChar;
begin
  if BeginText then
    Result := PdfText_GetUnicode(FTextHandle, CharIndex)
  else
    Result := #0;
end;

function TPdfPage.GetCharFontSize(CharIndex: Integer): Double;
begin
  if BeginText then
    Result := PdfText_GetFontSize(FTextHandle, CharIndex)
  else
    Result := 0;
end;

function TPdfPage.GetCharBox(CharIndex: Integer): TPdfRect;
begin
  if BeginText then
    PdfText_GetCharBox(FTextHandle, CharIndex, Result.Left, Result.Right, Result.Bottom, Result.Top)
  else
    EmptyPdfRect(Result);
end;

function TPdfPage.GetCharIndexAt(PageX, PageY, ToleranceX, ToleranceY: Double): Integer;
begin
  if BeginText then
    Result := PdfText_GetCharIndexAtPos(FTextHandle, PageX, PageY, ToleranceX, ToleranceY)
  else
    Result := 0;
end;

function TPdfPage.ReadText(CharIndex, Count: Integer): Widestring;
var
  Len: Integer;
begin
  if (Count > 0) and BeginText then
  begin
    SetLength(Result, Count);
    Len := PdfText_GetText(FTextHandle, CharIndex, Count + 1, PWideChar(Result)); // include #0 terminater
    if Len = 0 then
      Result := ''
    else if Len + 1 < Count then
      SetLength(Result, Len - 1);
  end
  else
    Result := '';
end;

function TPdfPage.GetTextAt(Left, Top, Right, Bottom: Double): Widestring;
var
  Len: Integer;
begin
  if BeginText then
  begin
    Len := PdfText_GetBoundedText(FTextHandle, Left, Top, Right, Bottom, nil, 0); // excluding #0 terminator
    SetLength(Result, Len);
    if Len > 0 then
      PdfText_GetBoundedText(FTextHandle, Left, Top, Right, Bottom, PWideChar(Result), Len);
  end
  else
    Result := '';
end;

function TPdfPage.GetTextAt(const R: TPdfRect): Widestring;
begin
  Result := GetTextAt(R.Left, R.Top, R.Right, R.Bottom);
end;

function TPdfPage.GetTextRectCount(CharIndex, Count: Integer): Integer;
begin
  if BeginText then
    Result := PdfText_CountRects(FTextHandle, CharIndex, Count)
  else
    Result := 0;
end;

function TPdfPage.GetTextRect(RectIndex: Integer): TPdfRect;
begin
  if BeginText then
    PdfText_GetRect(FTextHandle, RectIndex, Result.Left, Result.Top, Result.Right, Result.Bottom)
  else
    EmptyPdfRect(Result);
end;

function TPdfPage.GetWebLinkCount: Integer;
begin
  if BeginWebLinks then
  begin
    Result := PdfLink_CountWebLinks(FLinkHandle);
    if Result < 0 then
      Result := 0;
  end
  else
    Result := 0;
end;

function TPdfPage.GetWebLinkURL(LinkIndex: Integer): Widestring;
var
  Len: Integer;
begin
  Result := '';
  if BeginWebLinks then
  begin
    Len := PdfLink_GetURL(FLinkHandle, LinkIndex, nil, 0) - 1; // including #0 terminator
    if Len > 0 then
    begin
      SetLength(Result, Len);
      PdfLink_GetURL(FLinkHandle, LinkIndex, PWideChar(Result), Len + 1); // including #0 terminator
    end;
  end;
end;

function TPdfPage.GetWebLinkRectCount(LinkIndex: Integer): Integer;
begin
  if BeginWebLinks then
    Result := PdfLink_CountRects(FLinkHandle, LinkIndex)
  else
    Result := 0;
end;

function TPdfPage.GetWebLinkRect(LinkIndex, RectIndex: Integer): TPdfRect;
begin
  if BeginWebLinks then
    PdfLink_GetRect(FLinkHandle, LinkIndex, RectIndex, Result.Left, Result.Top, Result.Right, Result.Bottom)
  else
    EmptyPdfRect(Result);
end;

{ _TPdfBitmapHideCtor }

procedure _TPdfBitmapHideCtor.Create;
begin
  inherited Create;
end;

{ TPdfBitmap }

constructor TPdfBitmap.Create(ABitmap: FPDF_BITMAP; AOwnsBitmap: Boolean);
begin
  inherited Create;
  FBitmap := ABitmap;
  FOwnsBitmap := AOwnsBitmap;
  if FBitmap <> nil then
  begin
    FWidth := PdfBitmap_GetWidth(FBitmap);
    FHeight := PdfBitmap_GetHeight(FBitmap);
    FBytesPerScanLine := PdfBitmap_GetStride(FBitmap);
  end;
end;

constructor TPdfBitmap.Create(AWidth, AHeight: Integer; AAlpha: Boolean);
begin
  Create(PdfBitmap_Create(AWidth, AHeight, Ord(AAlpha)), True);
end;

constructor TPdfBitmap.Create(AWidth, AHeight: Integer; AFormat: TPdfBitmapFormat);
begin
  Create(PdfBitmap_CreateEx(AWidth, AHeight, Ord(AFormat), nil, 0), True);
end;

constructor TPdfBitmap.Create(AWidth, AHeight: Integer; AFormat: TPdfBitmapFormat; ABuffer: Pointer;
  ABytesPerScanLine: Integer);
begin
  Create(PdfBitmap_CreateEx(AWidth, AHeight, Ord(AFormat), ABuffer, ABytesPerScanline), True);
end;

destructor TPdfBitmap.Destroy;
begin
  if FOwnsBitmap and (FBitmap <> nil) then
    PdfBitmap_Destroy(FBitmap);
  inherited Destroy;
end;

function TPdfBitmap.GetBuffer: Pointer;
begin
  if FBitmap <> nil then
    Result := PdfBitmap_GetBuffer(FBitmap)
  else
    Result := nil;
end;

procedure TPdfBitmap.FillRect(ALeft, ATop, AWidth, AHeight: Integer; AColor: FPDF_DWORD);
begin
  if FBitmap <> nil then
    PdfBitmap_FillRect(FBitmap, ALeft, ATop, AWidth, AHeight, AColor);
end;

{ TPdfPoint }

procedure TPdfPoint.Offset(XOffset, YOffset: Double);
begin
  X := X + XOffset;
  Y := Y + YOffset;
end;

class function TPdfPoint.Empty: TPdfPoint;
begin
  Result.X := 0;
  Result.Y := 0;
end;

initialization
  InitializeCriticalSectionAndSpinCount(PDFiumInitCritSect, 4000);

finalization
  DeleteCriticalSection(PDFiumInitCritSect);

end.
