{ Envision test cases

  Note:

  It took a while to figure but somehow FastMM would raise access to
  object after being freed in the most basic case when freeing
  a TMemoryStream. Still after reducing a test case to the most basic the
  error could sometimes occur.

  With NeverSleepOnThreadContention defined in FastMM4Options.inc the
  error does not occur.

}
unit TestCases;

{$M+}

{$I Envision.Inc}

interface

uses
    Classes,          { for TMemoryStream }
    EnDiGrph,         { for TDibGraphicClass }
    Graphics,         { for TGraphic }
    EnMisc,           { for TImageFormat }
    TestFrameWork;



type

{$ifdef __Delphi2010AndAbove}
 {$RTTI EXPLICIT METHODS([vcPublished]) PROPERTIES([vcPublished])}
{$endif}

TEnvisionTestCase = class(TTestCase)
protected
     function CreateGraphicFromStream( const Stream               : TStream;
                                       const ExpectedGraphicClass : TDibGraphicClass;
                                       const ExpectedImageCount   : Integer
                                     ) : TDibGraphic;

     function CreateGraphicFromFile( const FileName             : String;
                                     const ExpectedGraphicClass : TDibGraphicClass;
                                     const ExpectedImageCount   : Integer
                                   ) : TDibGraphic;

     procedure Convert( const Graphic     : TDibGraphic;
                        const ImageFormat : TImageFormat );

end;

TEnvisionBaseTest = class(TEnvisionTestCase)
published
    procedure TestRegisterUnregister;

end;

TEnvisionAssistedTest = class(TEnvisionTestCase)
protected
    FPagesScanned : Integer;

    procedure ReceiveImageFromScanner( const DibHandle    : THandle;
                                       const XDpi         : Word;
                                       const YDpi         : Word;
                                       const CallBackData : Pointer );

published
    procedure TestTwainScanning;
end;

TImageReadTest = class(TEnvisionTestCase)
 published

   procedure TestBmp;
   procedure TestDcx;
   procedure TestDcxMultiPage;
   procedure TestDds;
   procedure TestEps;
   procedure TestFaxG3;
   procedure TestIco;
   procedure TestJ2k;
   procedure TestJng;
   procedure TestJp2;
   procedure TestJpg;
   procedure TestMng;
   procedure TestPict;
   procedure TestPcx;
   procedure TestPdf;
   procedure TestPdfMultiPage;
   procedure TestPng;
   procedure TestPpm;
   procedure TestPsd;
   procedure TestTga;
   procedure TestTiff;
   procedure TestTiffMultiPage;
   procedure TestWbmp;
   procedure TestWmf;
   procedure TestXbm;
   procedure TestXpm;

 end;

TImageWriteTest = class(TEnvisionTestCase)
 protected
     { used in MultiSaveToStream }
     FPageToSave : TDibGraphic;
     function GetPage( const ImageNo     : Integer;
                       var   FreeGraphic : Boolean ) : TGraphic;


 published

   procedure TestBmp;
   procedure TestDcx;
   procedure TestDcxMultiPage;
   procedure TestDds;
   procedure TestEps;
   procedure TestFaxG3;
   procedure TestIco;
   procedure TestJ2k;
   procedure TestJng;
   procedure TestJp2;
   procedure TestJpg;
   procedure TestMng;
   procedure TestPict;
   procedure TestPcx;
   procedure TestPdf;
   procedure TestPdfMultiPage;
   procedure TestPng;
   procedure TestPpm;
   procedure TestPsd;
   procedure TestTga;
   procedure TestTiff;
   procedure TestTiffMultiPage;
   procedure TestWbmp;
   procedure TestWmf;
   procedure TestXbm;
   procedure TestXpm;

 end;


procedure SetTestImagesFolder( const Folder : String );

procedure SetPrivateTestImagesFolder( const Folder : String );

implementation


uses
    {$ifdef __DelphiXE7AndAbove}
    System.UITypes, { prevent compiler warning }
    {$endif}

    EnPdfium,  { for pdfGray }
    SysUtils,  { for fmOpenReadOnly }
    Windows,   { to prevent Inline expansion warning }
    Dialogs,   { for MessageDlg }
    Forms,     { for TScreen }
    EnTransf,  { for TImageFormatTransform }
    EnReg,     { to register TDibGraphic classes }
    EnScan,    { for TScanner }

    EnWBmpGr,
    EnFaxG3Gr,
    EnXpmGr,
    EnXbmGr,
    EnPictGr,
    EnJ2kGr,
    EnJp2Gr,
    EnJngGr,
    EnMngGr,
    EnDdsGr,
    EnEpsGr,
    EnPdfGr,
    EnTifGr,
    EnPpmGr,
    EnIcoGr,
    EnWmfGr,
    EnDcxGr,
    EnJpgGr,
    EnTgaGr,
    EnPngGr,
    EnPsdGr,
    EnPcxGr,
    EnBmpGr;

var

TestImagesFolder        : String;
PrivateTestImagesFolder : String;

procedure SetTestImagesFolder( const Folder : String );
begin
    TestImagesFolder := Folder;
end;

procedure SetPrivateTestImagesFolder( const Folder : String );
begin
    PrivateTestImagesFolder := Folder;
end;

{--------------------------------------------------------------------------}

{ returns True if pixel primarily dark }
function IsPixelDark( const Graphic   : TDibGraphic;
                      const X         : LongInt;
                      const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Red <= 128) and
              (Graphic.Rgb[X,Y].Green <= 128) and
              (Graphic.Rgb[X,Y].Blue <= 128);

end;

{--------------------------------------------------------------------------}

{ returns True if pixel primarily light }
function IsPixelLight( const Graphic   : TDibGraphic;
                       const X         : LongInt;
                       const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Red > 128) and
              (Graphic.Rgb[X,Y].Green > 128) and
              (Graphic.Rgb[X,Y].Blue > 128);
end;

{--------------------------------------------------------------------------}

{ returns True if pixel primarily red }
function IsPixelRed( const Graphic   : TDibGraphic;
                     const X         : LongInt;
                     const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Red > Graphic.Rgb[X,Y].Blue + 30) and
              (Graphic.Rgb[X,Y].Red > Graphic.Rgb[X,Y].Green + 30);
end;

{--------------------------------------------------------------------------}

{ returns True if pixel primarily green }
function IsPixelGreen( const Graphic   : TDibGraphic;
                       const X         : LongInt;
                       const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Green > Graphic.Rgb[X,Y].Blue + 30) and
              (Graphic.Rgb[X,Y].Green > Graphic.Rgb[X,Y].Red + 30);
end;

{--------------------------------------------------------------------------}

{ returns True if pixel primarily blue }
function IsPixelBlue( const Graphic   : TDibGraphic;
                      const X         : LongInt;
                      const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Blue > Graphic.Rgb[X,Y].Green + 30) and
              (Graphic.Rgb[X,Y].Blue > Graphic.Rgb[X,Y].Red + 30);
end;

{--------------------------------------------------------------------------}

function IsPixelWhite( const Graphic   : TDibGraphic;
                       const X         : LongInt;
                       const Y         : LongInt ) : Boolean;
begin
    Result := (Graphic.Rgb[X,Y].Red > 250) and
              (Graphic.Rgb[X,Y].Green > 250) and
              (Graphic.Rgb[X,Y].Blue > 250);
end;

{--------------------------------------------------------------------------}

function TestImageFileName( const Name : String ) : String;
begin
    Result := TestImagesFolder + Name;
end;

{--------------------------------------------------------------------------}

function PrivateTestImageFileName( const Name : String ) : String;
begin
    Result := PrivateTestImagesFolder + Name;
end;

{--------------------------------------------------------------------------}

function TempTestImageFileName( const Ext : String ) : String;
begin
    Result := EnMisc.GetTempFileName( 'Img', Ext );
end;

{--------------------------------------------------------------------------}


procedure TEnvisionBaseTest.TestRegisterUnregister;
var
    Stream : TMemoryStream;

begin
    UnregisterDibGraphic( TTgaGraphic );

    Check( FindDibGraphicClass('tGa') = nil );
    Check( FindDibGraphicClass('tga') = nil );
    Check( FindDibGraphicClass('.afi') = nil );
    Check( FindDibGraphicClass('TGA') = nil );
    Check( FindDibGraphicClass('vsT') = nil );
    Check( FindDibGraphicClass('.tga') = nil );
    Check( FindDibGraphicClass(TestImageFileName( 'tigers.tga' )) = nil );

    Check( Pos( 'Targa', SaveAsGraphicFilter ) = 0 );
    Check( Pos( 'Vista', SaveAsGraphicFilter ) = 0 );

    RegisterDibGraphic('TGA', 'Truevision Targa', TTgaGraphic);
    RegisterDibGraphic('VST', 'Truevision Vista', TTgaGraphic);
    RegisterDibGraphic('AFI', 'Truevision Targa', TTgaGraphic);

    Check( FindDibGraphicClass('tGa') = TTgaGraphic );
    Check( FindDibGraphicClass('tga') = TTgaGraphic );
    Check( FindDibGraphicClass('.afi') = TTgaGraphic );
    Check( FindDibGraphicClass('TGA') = TTgaGraphic );
    Check( FindDibGraphicClass('vsT') = TTgaGraphic );
    Check( FindDibGraphicClass('.tga') = TTgaGraphic );
    Check( FindDibGraphicClass(TestImageFileName( 'tigers.tga' )) = TTgaGraphic );

    Check( Pos( 'Targa', SaveAsGraphicFilter ) > 0 );
    Check( Pos( 'Vista', SaveAsGraphicFilter ) > 0 );

    Check( GuessDibGraphicClass(TestImageFileName( 'tigers.tga' )) = TTgaGraphic );

    Stream := TMemoryStream.Create;
    try
        Stream.LoadFromFile(TestImageFileName( 'tigers.tga' ));
        Check( GuessDibGraphicClass(Stream) = TTgaGraphic );
    finally
        Stream.Free;
    end;
end;

{--------------------------------------------------------------------------}

function TEnvisionTestCase.CreateGraphicFromFile(
                                     const FileName             : String;
                                     const ExpectedGraphicClass : TDibGraphicClass;
                                     const ExpectedImageCount   : Integer
                                   ) : TDibGraphic;
{$J+}
const
    LastLoad : Integer = 0;
var
    GuessedClass   : TDibGraphicClass;

    SecondGraphic  : TDibGraphic;
    Stream         : TStream;
begin

    GuessedClass := GuessDibGraphicClass(FileName);
    Check( GuessedClass = ExpectedGraphicClass );

    Check( ExpectedGraphicClass.CanRead(FileName ) );

    Result := ExpectedGraphicClass.Create;
    Check( Result.GetImageCount( FileName ) = ExpectedImageCount );

    Result.LoadFromFile( FileName );

    { test same equivalent using stream based routines with different
    types of stream and sharing }
    Stream        := nil;
    SecondGraphic := nil;
    try
        case (LastLoad mod 3) of

            0:
            begin
               Stream := TMemoryStream.Create;
               TMemoryStream(Stream).LoadFromFile( FileName );
            end;

            1:
            begin
                Stream := TFileStream.Create( FileName, fmOpenRead or fmShareDenyWrite );
            end;

            2:
            begin
                Stream := TFileStream.Create( FileName, fmOpenRead or fmShareExclusive );
            end;
        end;

        Inc(LastLoad);
        if LastLoad > 1000 then
           LastLoad := 0;

        Stream.Seek( 0, soFromBeginning );

        SecondGraphic := CreateGraphicFromStream( Stream, ExpectedGraphicClass, ExpectedImageCount );

        if not GraphicsIdentical(Result, SecondGraphic) then
            raise Exception.Create('LoadFromStream and LoadFromFile not identical');
    finally
        Stream.Free;
        SecondGraphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

function TEnvisionTestCase.CreateGraphicFromStream(
                                       const Stream               : TStream;
                                       const ExpectedGraphicClass : TDibGraphicClass;
                                       const ExpectedImageCount   : Integer
                                     ) : TDibGraphic;
var
    GuessedClass : TDibGraphicClass;
begin

    GuessedClass := GuessDibGraphicClass(Stream);
    Check( GuessedClass = ExpectedGraphicClass );

    Check( ExpectedGraphicClass.CanRead(Stream ) );

    Result := ExpectedGraphicClass.Create;
    Check( Result.GetImageCount( Stream ) = ExpectedImageCount );

    Result.LoadFromStream( Stream );
end;

{--------------------------------------------------------------------------}

procedure TEnvisionTestCase.Convert(
                        const Graphic     : TDibGraphic;
                        const ImageFormat : TImageFormat );
var
    Transform : TImageFormatTransform;
begin
    Transform := TImageFormatTransform.Create;
    try
        Transform.ImageFormat := ImageFormat;
        Transform.Quantize    := True;
        Transform.Dither      := True;
        Transform.Apply(Graphic);
    finally
        Transform.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestBmp;
var
    Graphic : TDibGraphic;
    Stream  : TStream;
begin

    Graphic := nil;
    Stream  := TMemoryStream.Create;
    try

        TMemoryStream(Stream).LoadFromFile( TestImageFileName( 'tomcatTC.bmp') );
        Stream.Seek( 0, soFromBeginning );

        Graphic := CreateGraphicFromStream( Stream, TBitmapGraphic, 1 );

        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 141 );
        Check( Graphic.Height = 70 );
        Check( IsPixelLight( Graphic, 8, 8 ) );
        Check( IsPixelLight( Graphic, 130, 60 ) );
        Check( IsPixelDark( Graphic, 63, 25 ) );
        Check( IsPixelBlue( Graphic, 84, 44 ) );

    finally
        Stream.Free;
        Graphic.Free;
    end;

    { when loading from a file stream, when sharing is allowed the underlying
      FreeImage library will be more efficient }
    Graphic := nil;
    Stream  := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'tomcatTC.bmp'), fmOpenRead or fmShareDenyWrite );
        Stream.Seek( 0, soFromBeginning );
        Graphic := CreateGraphicFromStream( Stream, TBitmapGraphic, 1 );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 141 );
        Check( Graphic.Height = 70 );
        Check( IsPixelLight( Graphic, 8, 8 ) );
        Check( IsPixelLight( Graphic, 130, 60 ) );
        Check( IsPixelDark( Graphic, 63, 25 ) );
        Check( IsPixelBlue( Graphic, 84, 44 ) );
    finally
        Graphic.Free;
        Stream.Free;
    end;

    Stream  := nil;
    Graphic := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'tomcatTC.bmp'), fmOpenRead or fmShareExclusive );
        Stream.Seek( 0, soFromBeginning );
        Graphic := CreateGraphicFromStream( Stream, TBitmapGraphic, 1 );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 141 );
        Check( Graphic.Height = 70 );
        Check( IsPixelLight( Graphic, 8, 8 ) );
        Check( IsPixelLight( Graphic, 130, 60 ) );
        Check( IsPixelDark( Graphic, 63, 25 ) );
        Check( IsPixelBlue( Graphic, 84, 44 ) );
    finally
        Graphic.Free;
        Stream.Free;
    end;

    Graphic := CreateGraphicFromFile(
                            TestImageFileName('tomcatBW.bmp'),
                            TBitmapGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 141 );
        Check( Graphic.Height = 70 );
        Check( IsPixelLight( Graphic, 8, 8 ) );
        Check( IsPixelLight( Graphic, 130, 60 ) );
        Check( IsPixelDark( Graphic, 63, 25 ) );


        Graphic.LoadFromFile( TestImageFileName( 'tomcatTC.bmp'));
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 141 );
        Check( Graphic.Height = 70 );
        Check( IsPixelLight( Graphic, 8, 8 ) );
        Check( IsPixelLight( Graphic, 130, 60 ) );
        Check( IsPixelDark( Graphic, 63, 25 ) );
        Check( IsPixelBlue( Graphic, 84, 44 ) );

    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile(
                            TestImageFileName('test32.bmp'),
                            TBitmapGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1200 );
        Check( Graphic.Height = 1115);
        Check( IsPixelBlue( Graphic, 765, 1094 ) );
        Check( IsPixelBlue( Graphic, 331, 15 ) );
        Check( IsPixelRed( Graphic, 1123, 408 ) );
        Check( IsPixelRed( Graphic, 1106, 17 ) );
        Check( IsPixelGreen( Graphic, 1062, 920 ) );
        Check( IsPixelGreen( Graphic, 736, 21 ) );
    finally
        Graphic.Free;
    end;


end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPng;
var
    Graphic     : TDibGraphic;
    Color       : TRgb;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'pngwithbk.png' ), TPngGraphic, 1 );
    try
        { test by using the background color specified in the PNG file }
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 32 );
        Check( Graphic.Height = 32 );
        Check( IsPixelLight( Graphic, 1, 6 ) );
        Check( IsPixelLight( Graphic, 1, 28 ) );
        Check( not IsPixelGreen( Graphic, 1, 6 ) );
        Check( not IsPixelGreen( Graphic, 1, 28 ) );

        { test with a specific background color }
        Graphic.OverwriteFileBackground := True;
        Color.Red     := 0;
        Color.Green   := 255;
        Color.Blue    := 0;
        Graphic.BackgroundColor := Color;
        Graphic.LoadFromFile( TestImageFileName( 'pngwithbk.png' ) );
        Check( Graphic.Width = 32 );
        Check( Graphic.Height = 32 );
        Check( not IsPixelLight( Graphic, 1, 6 ) );
        Check( not IsPixelLight( Graphic, 1, 28 ) );
        Check( IsPixelGreen( Graphic, 1, 6 ) );
        Check( IsPixelGreen( Graphic, 1, 28 ) );

    finally
        Graphic.Free;
    end;

    { PNG file not properly handled by FreeImage, will fall back on GDI+ }
    if FileExists(PrivateTestImageFileName( 'test.png' )) then
    begin
        Graphic := CreateGraphicFromFile( PrivateTestImageFileName( 'test.png' ), TPngGraphic, 1 );
        try
            Check( Graphic.ImageFormat = ifTrueColor );
            Check( Graphic.Width = 1241 );
            Check( Graphic.Height = 1777 );
            Check( IsPixelLight( Graphic, 31, 955 ) );
            Check( IsPixelDark( Graphic, 25, 479 ) );
            Check( IsPixelLight( Graphic, 831, 946 ) );
            Check( IsPixelDark( Graphic, 875, 896 ) );
            Check( IsPixelRed( Graphic, 441, 1161 ) );
            Check( IsPixelRed( Graphic, 1026, 927 ) );

        finally
            Graphic.Free;
        end;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPsd;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.psd' ), TPsdGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 321 );
        Check( Graphic.Height = 236 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 310, 222 ) );
        Check( IsPixelDark( Graphic, 72, 160 ) );
        Check( IsPixelDark( Graphic, 246, 130 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPict;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'marbles.pct' ), TPictGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1419 );
        Check( Graphic.Height = 1001 );
        Check( IsPixelLight( Graphic, 731, 213 ) );
        Check( IsPixelLight( Graphic, 427, 650 ) );
        Check( IsPixelBlue( Graphic, 670, 82 ) );
        Check( IsPixelGreen( Graphic, 1337, 653 ) );
        Check( IsPixelRed( Graphic, 757, 938 ) );

    finally
        Graphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPcx;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigerstc.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers256c.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifColor256 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers256g.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifGray256 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers16c.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifColor16 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers16g.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifGray16 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigersbw.pcx' ), TPcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPdf;
var
    Graphic     : TPdfGraphic;
begin

    Graphic := TPdfGraphic.Create;
    try
        Graphic.LoadFromFile( TestImageFileName( 'tigers.pdf' ) );

        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1605 );
        Check( Graphic.Height = 1180 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 1549, 1126 ) );
        Check( IsPixelDark( Graphic, 158, 941 ) );
        Check( IsPixelDark( Graphic, 1224, 648 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPdfMultiPage;
var
    Graphic     : TPdfGraphic;
    Stream      : TStream;
begin

    Graphic := TPdfGraphic.Create;
    try
        Check( Graphic.GetImageCountFromFile( TestImageFileName('tigers.pdf') ) = 2 );

        Graphic.ImageToLoad := 1;
        Graphic.LoadFromFile( TestImageFileName( 'tigers.pdf' ) );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1605 );
        Check( Graphic.Height = 1180 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 1549, 1126 ) );
        Check( IsPixelDark( Graphic, 158, 941 ) );
        Check( IsPixelDark( Graphic, 1224, 648 ) );

        Graphic.ImageToLoad := 2;
        Graphic.LoadFromFile( TestImageFileName( 'tigers.pdf' ) );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1538 );
        Check( Graphic.Height = 1138 );
        Check( IsPixelLight( Graphic, 1075, 219 ) );
        Check( IsPixelLight( Graphic, 935, 1019 ) );
        Check( IsPixelDark( Graphic, 213, 550 ) );
        Check( IsPixelDark( Graphic, 1172, 625 ) );

    finally
        Graphic.Free;
    end;

    { multipage load from memory stream }
    Graphic := TPdfGraphic.Create;
    Stream  := nil;
    try
        Stream := TMemoryStream.Create;
        TMemoryStream(Stream).LoadFromFile( TestImageFileName('tigers.pdf') );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1538 );
        Check( Graphic.Height = 1138 );
        Check( IsPixelLight( Graphic, 1075, 219 ) );
        Check( IsPixelLight( Graphic, 935, 1019 ) );
        Check( IsPixelDark( Graphic, 213, 550 ) );
        Check( IsPixelDark( Graphic, 1172, 625 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    { multipage load from file stream opened exclusive }
    Graphic := TPdfGraphic.Create;
    Stream  := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'tigers.pdf'), fmOpenRead or fmShareExclusive );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1538 );
        Check( Graphic.Height = 1138 );
        Check( IsPixelLight( Graphic, 1075, 219 ) );
        Check( IsPixelLight( Graphic, 935, 1019 ) );
        Check( IsPixelDark( Graphic, 213, 550 ) );
        Check( IsPixelDark( Graphic, 1172, 625 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    { multipage load from file stream opened shared }
    Graphic := TPdfGraphic.Create;
    Stream  := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'tigers.pdf'), fmOpenRead or fmShareDenyWrite );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1538 );
        Check( Graphic.Height = 1138 );
        Check( IsPixelLight( Graphic, 1075, 219 ) );
        Check( IsPixelLight( Graphic, 935, 1019 ) );
        Check( IsPixelDark( Graphic, 213, 550 ) );
        Check( IsPixelDark( Graphic, 1172, 625 ) );

    finally
        Stream.Free;
        Graphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestTga;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.tga' ), TTgaGraphic, 1 );
    try
        Graphic.LoadFromFile( TestImageFileName( 'tigers.tga' ) );
        Check( Graphic.ImageFormat = ifColor256 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    { test with tga file that doesn't have tga extension }
    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.exttga' ), TTgaGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifColor256 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestJ2k;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'bretagne2.j2k' ), TJ2kGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 2592 );
        Check( Graphic.Height = 1944 );
        Check( IsPixelBlue( Graphic, 10, 10 ) );
        Check( IsPixelBlue( Graphic, 2300, 200 ) );
        Check( IsPixelDark( Graphic, 2590, 1942 ) );
        Check( IsPixelDark( Graphic, 2, 1942 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestJng;
var
    Graphic     : TDibGraphic;
    Color       : TRgb;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.jng' ), TJngGraphic, 1 );
    try
        { test by using the default background color }
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelWhite( Graphic, 10, 10 ) );
        Check( IsPixelWhite( Graphic, 630, 460) );
        Check( IsPixelDark( Graphic, 142, 320 ) );
        Check( IsPixelDark( Graphic, 489, 260 ) );
        Check( IsPixelLight( Graphic, 447, 90 ) );


        { test with a specific background color }
        Graphic.OverwriteFileBackground := True;
        Color.Red     := 0;
        Color.Green   := 255;
        Color.Blue    := 0;
        Graphic.BackgroundColor := Color;
        Graphic.LoadFromFile( TestImageFileName( 'tigers.jng' ) );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelGreen( Graphic, 10, 10 ) );
        Check( IsPixelGreen( Graphic, 630, 460) );
        Check( IsPixelDark( Graphic, 142, 320 ) );
        Check( IsPixelDark( Graphic, 489, 260 ) );
        Check( IsPixelLight( Graphic, 447, 90 ) );

    finally
        Graphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestJp2;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.jp2' ), TJp2Graphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;
end;


{--------------------------------------------------------------------------}

procedure TImageReadTest.TestJpg;
var
    Graphic     : TDibGraphic;
    Stream      : TMemoryStream;

begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.jpg' ), TJpegGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;

    {
      one user reported that loading this file raised an exception in
      FreeImage_LoadFromHandle with the DLL is loaded in memory

      The issue has been reproduced on Windows 8.

      When the free image DLL is in memory and on windows 8
      FreeImage_LoadFromHandle will raise a fatal AV that crashed the entire
      program. When free image is not loaded from memory, FreeImage_LoadFromHandle
      returns nil without crashing, and them internally the file is loaded
      with GDI+.

      It's a bit of a messy situation. The memory dll loading situation causes
      a crash, but in either case, free image is not able to load this file at
      all.

      When trying to load the file with KuView (an image viewer built with
      FreeImage, a message is displayed,

      Corrupt JPEG data: 66 extranous bytes before marker 0xd8.
      Invalid JPEG structure: two SOI markers

      This error gets "seen" in EnFreeImageAPI.ErrorHandler but the call
      to FreeImage_LoadFromHandle will crash application before returning.

      This situation will probably happen anytime the error handler is called
      in 64 bits, with DLL in memory, and using windows 8 or above.

      Now this has been reproduced under windows 7.

      So the only workaround I have found is to not load the DLL from memory
      if compiled under 64 bits.

      See EnFreeImageAPI.LoadDLL routine.
    }


    Graphic := CreateGraphicFromFile( TestImageFileName( '44.jpg' ), TJpegGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 2048 );
        Check( Graphic.Height = 1536 );
        Check( IsPixelLight( Graphic, 212, 1030 ) );
        Check( IsPixelDark( Graphic, 2028, 1172 ) );
        Check( IsPixelLight( Graphic, 2015, 827 ) );
        Check( IsPixelDark( Graphic, 855, 360 ) );
    finally
        Graphic.Free;
    end;

    {
      an empty file would previous raise AV and hang application in version 3.00. Fixed
      in 3.01

      Reported from a user:

      Although this is probably a rare case, if a file is opened with the EnJpgGr filter (and perhaps other En_ filters)
      if you have a zero byte file with a .jpg extension (which kicks in your code), on Windows 8/2012 it cases a
      complete melt down of the app (i.e. Project1.exe has stopped responding, check online, close, debug)
      This does not seem to occur with the same app, image file etc on Windows 7.
    }

    { test with empty stream }
    Stream  := nil;
    Graphic := nil;
    try
        Stream := TMemoryStream.Create;
        Graphic := TJpegGraphic.Create;

        Graphic.LoadFromStream( Stream );
        Check( Graphic.Width = 0 );
        Check( Graphic.Height = 0 );

    finally
        Graphic.Free;
        Stream.Free;
    end;

    { test with position at end of stream (like and empty stream) }
    Stream  := nil;
    Graphic := nil;
    try
        Stream := TMemoryStream.Create;
        Stream.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Graphic := TJpegGraphic.Create;

        Stream.Seek( 0, soFromEnd );
        Graphic.LoadFromStream( Stream );
        Check( Graphic.Width = 0 );
        Check( Graphic.Height = 0 );

    finally
        Graphic.Free;
        Stream.Free;
    end;

    { test with an empty file }
    Graphic := TJpegGraphic.Create;
    try
        Graphic.LoadFromFile( TestImageFileName( 'empty.jpg' ) );
        Check( Graphic.Width = 0 );
        Check( Graphic.Height = 0 );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestMng;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.mng' ), TMngGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 492 );
        Check( Graphic.Height = 364 );
        Check( IsPixelLight( Graphic, 344, 70 ) );
        Check( IsPixelLight( Graphic, 299, 326 ) );
        Check( IsPixelDark( Graphic, 68, 176 ) );
        Check( IsPixelDark( Graphic, 375, 200 ) );

    finally
        Graphic.Free;
    end;

end;


{--------------------------------------------------------------------------}

procedure TImageReadTest.TestDcx;
var
    Graphic     : TDibGraphic;
begin
    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.dcx' ), TDcxGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 9, 9 ) );
        Check( IsPixelLight( Graphic, 619, 450 ) );
        Check( IsPixelDark( Graphic, 63, 376 ) );
        Check( IsPixelDark( Graphic, 489, 259 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestDcxMultiPage;
var
    Graphic     : TDcxGraphic;
    Stream      : TStream;
begin

    if FileExists(PrivateTestImageFileName('82236.dcx')) then
    begin
        Graphic := TDcxGraphic.Create;
        try
            Check( Graphic.GetImageCountFromFile( PrivateTestImageFileName('82236.dcx') ) = 2 );

            Graphic.ImageToLoad := 2;
            Graphic.LoadFromFile( PrivateTestImageFileName( '82236.dcx' ) );
            Check( Graphic.ImageFormat = ifBlackWhite );
            Check( Graphic.Width = 1252 );
            Check( Graphic.Height = 1024 );
            Check( IsPixelLight( Graphic, 40, 10 ) );
            Check( IsPixelLight( Graphic, 875, 734 ) );
            Check( IsPixelDark( Graphic, 702, 202 ) );
            Check( IsPixelDark( Graphic, 187, 949 ) );
        finally
            Graphic.Free;
        end;
    end;

    { multipage load from memory stream }
    if FileExists(PrivateTestImageFileName('82236.dcx')) then
    begin
        Graphic := TDcxGraphic.Create;
        Stream  := nil;
        try
            Stream := TMemoryStream.Create;
            TMemoryStream(Stream).LoadFromFile( PrivateTestImageFileName('82236.dcx') );
            Stream.Seek( 0, soFromBeginning );
            Check( Graphic.GetImageCount( Stream ) = 2 );

            Stream.Seek( 0, soFromBeginning );
            Graphic.ImageToLoad := 2;
            Graphic.LoadFromStream( Stream );
            Check( Graphic.ImageFormat = ifBlackWhite );
            Check( Graphic.Width = 1252 );
            Check( Graphic.Height = 1024 );
            Check( IsPixelLight( Graphic, 40, 10 ) );
            Check( IsPixelLight( Graphic, 875, 734 ) );
            Check( IsPixelDark( Graphic, 702, 202 ) );
            Check( IsPixelDark( Graphic, 187, 949 ) );
        finally
            Stream.Free;
            Graphic.Free;
        end;

        { multipage load from file stream opened exclusive }
        Graphic := TDcxGraphic.Create;
        Stream  := nil;
        try
            Stream := TFileStream.Create( PrivateTestImageFileName( '82236.dcx'), fmOpenRead or fmShareExclusive );
            Stream.Seek( 0, soFromBeginning );
            Check( Graphic.GetImageCount( Stream ) = 2 );

            Stream.Seek( 0, soFromBeginning );
            Graphic.ImageToLoad := 2;
            Graphic.LoadFromStream( Stream );
            Check( Graphic.ImageFormat = ifBlackWhite );
            Check( Graphic.Width = 1252 );
            Check( Graphic.Height = 1024 );
            Check( IsPixelLight( Graphic, 40, 10 ) );
            Check( IsPixelLight( Graphic, 875, 734 ) );
            Check( IsPixelDark( Graphic, 702, 202 ) );
            Check( IsPixelDark( Graphic, 187, 949 ) );
        finally
            Stream.Free;
            Graphic.Free;
        end;

        { multipage load from file stream opened shared }
        Graphic := TDcxGraphic.Create;
        Stream  := nil;
        try
            Stream := TFileStream.Create( PrivateTestImageFileName( '82236.dcx'), fmOpenRead or fmShareDenyWrite );
            Stream.Seek( 0, soFromBeginning );
            Check( Graphic.GetImageCount( Stream ) = 2 );

            Stream.Seek( 0, soFromBeginning );
            Graphic.ImageToLoad := 2;
            Graphic.LoadFromStream( Stream );
            Check( Graphic.ImageFormat = ifBlackWhite );
            Check( Graphic.Width = 1252 );
            Check( Graphic.Height = 1024 );
            Check( IsPixelLight( Graphic, 40, 10 ) );
            Check( IsPixelLight( Graphic, 875, 734 ) );
            Check( IsPixelDark( Graphic, 702, 202 ) );
            Check( IsPixelDark( Graphic, 187, 949 ) );

        finally
            Stream.Free;
            Graphic.Free;
        end;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestDds;
var
    Graphic     : TDibGraphic;
begin
    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.dds' ), TDdsGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 492 );
        Check( Graphic.Height = 364 );
        Check( IsPixelLight( Graphic, 344, 70 ) );
        Check( IsPixelLight( Graphic, 299, 326 ) );
        Check( IsPixelDark( Graphic, 68, 176 ) );
        Check( IsPixelDark( Graphic, 375, 200 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestEps;
var
    Graphic : TDibGraphic;
begin
    Graphic := CreateGraphicFromFile( TestImageFileName( 'celebrat.eps' ), TEpsGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 204 );
        Check( Graphic.Height = 184 );
        Check( IsPixelLight( Graphic, 10, 10 ) );
        Check( IsPixelLight( Graphic, 82, 107 ) );
        Check( IsPixelDark( Graphic, 150, 156 ) );
        Check( IsPixelDark( Graphic, 15, 59 ) );
    finally
        Graphic.Free;
    end;


    Graphic := CreateGraphicFromFile( TestImageFileName( 'square.eps' ), TEpsGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifColor256 );
        Check( Graphic.Width = 178 );
        Check( Graphic.Height = 178 );
        Check( IsPixelLight( Graphic, 162, 162 ) );
        Check( IsPixelDark( Graphic, 80, 80 ) );
    finally
        Graphic.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestFaxG3;
var
    Graphic     : TFaxG3Graphic;
begin
    { fax g3 does not have signature so need to use  TFaxG3Graphic explicitly }
    Graphic := TFaxG3Graphic.Create;
    try
        Graphic.LoadFromFile( TestImageFileName( 'example.g3' ) );
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1728 );
        Check( Graphic.Height = 106 );
        Check( IsPixelLight( Graphic, 99, 55 ) );
        Check( IsPixelLight( Graphic, 1697, 45 ) );
        Check( IsPixelDark( Graphic, 112, 51 ) );
        Check( IsPixelDark( Graphic, 58, 60 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}


procedure TImageReadTest.TestIco;
var
    Graphic     : TIconGraphic;
begin

    Graphic := TIconGraphic.Create;
    try
        Graphic.BackgroundColor := MakeRGB( 0, 255, 0 );
        Graphic.LoadFromFile( TestImageFileName( 'image.ico' ) );
        Check( Graphic.ImageFormat = ifTrueColor );

        { icon size may be sensitive to windows display "scaling" }
        Check( Graphic.Width > 10 );
        Check( Graphic.Width < 200 );
        Check( Graphic.Width = Graphic.Height );


        if Graphic.Width = 32 then
        begin
            Check( Graphic.Width = 32 );
            Check( Graphic.Height = 32 );

            Check( IsPixelGreen( Graphic, 1, 1 ) );
            Check( IsPixelGreen( Graphic, 32, 32 ) );
            Check( IsPixelRed( Graphic, 15, 18 ) );
            Check( IsPixelDark( Graphic, 24, 24 ) );
        end;

    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestPpm;
var
    Graphic     : TDibGraphic;
begin

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.pbm' ), TPpmGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 45, 35 ) );
        Check( IsPixelLight( Graphic, 337, 370 ) );
        Check( IsPixelDark( Graphic, 137, 327 ) );
        Check( IsPixelDark( Graphic, 487, 258 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.pgm' ), TPpmGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifGray256 );
        Check( Graphic.Width = 257 );
        Check( Graphic.Height = 189 );
        Check( IsPixelLight( Graphic, 18, 10 ) );
        Check( IsPixelLight( Graphic, 180, 36 ) );
        Check( IsPixelDark( Graphic, 24, 151 ) );
        Check( IsPixelDark( Graphic, 187, 51 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tigers.ppm' ), TPpmGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 257 );
        Check( Graphic.Height = 189 );
        Check( IsPixelLight( Graphic, 18, 10 ) );
        Check( IsPixelLight( Graphic, 180, 36 ) );
        Check( IsPixelDark( Graphic, 24, 151 ) );
        Check( IsPixelDark( Graphic, 187, 51 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestTiff;
var
    Stream          : TMemoryStream;
    Graphic         : TDibGraphic;
    Position        : Int64;

    TempStream      : TMemoryStream;

begin
    if FileExists( PrivateTestImageFileName( '11007.tif' ) ) then
    begin
        Graphic := CreateGraphicFromFile( PrivateTestImageFileName( '11007.tif' ), TTiffGraphic, 1 );
        try
            Check( Graphic.ImageFormat = ifTrueColor );
            Check( Graphic.Width = 816 );
            Check( Graphic.Height = 1056 );
            Check( IsPixelRed( Graphic, 450, 58 ) );
            Check( IsPixelDark( Graphic, 732, 324 ) );
            Check( IsPixelDark( Graphic, 284, 997 ) );
            Check( IsPixelLight( Graphic, 384, 997 ) );
        finally
            Graphic.Free;
        end;
    end;

    { there was an old note in the initialization of EnPdfium that  without changing math flags this causes exception.
      Althoug this is a rare tif variation and page 1 is not rendered properly, in version 4.05 without changing the
      exception flags, there is no exception }
    if FileExists( PrivateTestImageFileName( 'TigersMathException.tif' ) ) then
    begin
        Graphic := CreateGraphicFromFile( PrivateTestImageFileName( 'TigersMathException.tif' ), TTiffGraphic, 2 );
        try
            Check( Graphic.ImageFormat = ifGray256 );
            Check( Graphic.Width = 256 );
            Check( Graphic.Height = 188 );

        finally
            Graphic.Free;
        end;
    end;


    { test 256 color }
    Stream  := nil;
    Graphic := nil;
    try
        Stream := TMemoryStream.Create;
        Stream.LoadFromFile( TestImageFileName( '153.tif' ) );
        Stream.Seek( 0, soFromBeginning );

        Graphic := CreateGraphicFromStream( Stream, TTiffGraphic, 1 );

        Check( Graphic.ImageFormat = ifColor256 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( Graphic.Width = 288 );
        Check( Graphic.Height = 216 );
        Check( IsPixelLight( Graphic, 52, 112 ) );
        Check( IsPixelDark( Graphic, 188, 124 ) );
        Check( IsPixelDark( Graphic, 89, 57 ) );
        Check( IsPixelRed( Graphic, 213, 56 ) );
        Check( IsPixelRed( Graphic, 200, 105 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    Stream  := nil;
    Graphic := nil;
    try
        Stream := TMemoryStream.Create;
        Stream.LoadFromFile( TestImageFileName( 'tigersbw.tif' ) );
        Stream.Seek( 0, soFromBeginning );

        Graphic := CreateGraphicFromStream( Stream, TTiffGraphic, 1 );

        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 219, 388 ) );
        Check( IsPixelLight( Graphic, 637, 351 ) );
        Check( IsPixelDark( Graphic, 576, 379 ) );
        Check( IsPixelDark( Graphic, 153, 258 ) );
        Check( (Graphic as TTiffGraphic).SoftwareTag = 'Envision Image Library' );
        Check( (Graphic as TTiffGraphic).DateTimeTag = '2013:06:26 20:37:06' );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    { test loading from stream, but not at beginning of stream }
    Stream     := nil;
    Graphic    := nil;
    TempStream := nil;
    try
        Stream := TMemoryStream.Create;

        { bytes before the tiff stream }
        Stream.LoadFromFile( TestImageFileName( 'tigers.pgm' ) );
        Stream.Seek( 0, soFromEnd );

        Position   := Stream.Position;

        TempStream := TMemoryStream.Create;
        TempStream.LoadFromFile( TestImageFileName( 'tigersbw.tif' ) );

        Stream.Write( TempStream.Memory^, TempStream.Size );

        Stream.Seek( Position, soFromBeginning );

        Graphic := CreateGraphicFromStream( Stream, TTiffGraphic, 1 );

        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( Graphic.Width = 641 );
        Check( Graphic.Height = 472 );
        Check( IsPixelLight( Graphic, 219, 388 ) );
        Check( IsPixelLight( Graphic, 637, 351 ) );
        Check( IsPixelDark( Graphic, 576, 379 ) );
        Check( IsPixelDark( Graphic, 153, 258 ) );
        Check( (Graphic as TTiffGraphic).SoftwareTag = 'Envision Image Library' );
        Check( (Graphic as TTiffGraphic).DateTimeTag = '2013:06:26 20:37:06' );

    finally
        Stream.Free;
        Graphic.Free;
        TempStream.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt4\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup4 );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;



    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\ccitt-rle\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcGroup3_1d );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;



    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg6\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;



    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\jpeg7\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcJPEG );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;


    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-bw\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelDark( Graphic, 40, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;



    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\lzw-color\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcLZW );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;



    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation1.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation2.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation3.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation4.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation5.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation6.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation7.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;

    Graphic := CreateGraphicFromFile( TestImageFileName( 'tiff-orientation\packbits-color\ExifOrientation8.tif' ), TTiffGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 1240 );
        Check( Graphic.Height = 1754 );
        Check( (Graphic as TTiffGraphic).Compression = tcPackbits );
        Check( IsPixelDark( Graphic, 636, 89 ) );
        Check( IsPixelRed( Graphic, 40, 170 ) );
        Check( IsPixelGreen( Graphic, 136, 170 ) );
        Check( IsPixelBlue( Graphic, 218, 170 ) );
        Check( IsPixelDark( Graphic, 89, 1662 ) );
        Check( IsPixelDark( Graphic, 5, 1735 ) );
        Check( IsPixelLight( Graphic, 1232, 1748 ) );
        Check( IsPixelLight( Graphic, 1232, 10 ) );
        Check( IsPixelLight( Graphic, 5, 5 ) );
    finally
        Graphic.Free;
    end;
    
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestTiffMultiPage;
var
    Graphic     : TTiffGraphic;
    Stream      : TStream;
    Position    : Int64;
    TempStream  : TMemoryStream;

begin
    Graphic := TTiffGraphic.Create;
    try
        Check( Graphic.GetImageCountFromFile( TestImageFileName('race.tif') ) = 2 );

        Graphic.ImageToLoad := 1;
        Graphic.LoadFromFile( TestImageFileName( 'race.tif' ) );
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 2585 );
        Check( Graphic.Height = 2011 );
        Check( IsPixelLight( Graphic, 411, 683 ) );
        Check( IsPixelLight( Graphic, 2021, 1235 ) );
        Check( IsPixelDark( Graphic, 2001, 1721 ) );
        Check( IsPixelDark( Graphic, 167, 515 ) );
    finally
        Graphic.Free;
    end;

    Graphic := TTiffGraphic.Create;
    try
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromFile( TestImageFileName( 'race.tif' ) );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 741 );
        Check( Graphic.Height = 570 );
        Check( IsPixelLight( Graphic, 186, 227 ) );
        Check( IsPixelLight( Graphic, 726, 539 ) );
        Check( IsPixelDark( Graphic, 553, 25 ) );
        Check( IsPixelDark( Graphic, 102, 537 ) );
        Check( IsPixelRed( Graphic, 92, 418 ) );
    finally
        Graphic.Free;
    end;

    { test loading from stream, but not at beginning of stream }
    Stream     := nil;
    Graphic    := nil;
    TempStream := nil;
    try
        Stream := TMemoryStream.Create;

        { bytes before the tiff stream }
        TMemoryStream(Stream).LoadFromFile( TestImageFileName( 'tigers.pgm' ) );
        Stream.Seek( 0, soFromEnd );

        Position   := Stream.Position;

        TempStream := TMemoryStream.Create;
        TempStream.LoadFromFile( TestImageFileName( 'race.tif' ) );

        Stream.Write( TempStream.Memory^, TempStream.Size );

        Stream.Seek( Position, soFromBeginning );

        Graphic := TTiffGraphic.Create;

        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( Position, soFromBeginning );

        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 741 );
        Check( Graphic.Height = 570 );
        Check( IsPixelLight( Graphic, 186, 227 ) );
        Check( IsPixelLight( Graphic, 726, 539 ) );
        Check( IsPixelDark( Graphic, 553, 25 ) );
        Check( IsPixelDark( Graphic, 102, 537 ) );
        Check( IsPixelRed( Graphic, 92, 418 ) );

    finally
        Stream.Free;
        Graphic.Free;
        TempStream.Free;
    end;

    { multipage load from memory stream }
    Graphic := TTiffGraphic.Create;
    Stream  := nil;
    try
        Stream := TMemoryStream.Create;
        TMemoryStream(Stream).LoadFromFile( TestImageFileName('race.tif') );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 741 );
        Check( Graphic.Height = 570 );
        Check( IsPixelLight( Graphic, 186, 227 ) );
        Check( IsPixelLight( Graphic, 726, 539 ) );
        Check( IsPixelDark( Graphic, 553, 25 ) );
        Check( IsPixelDark( Graphic, 102, 537 ) );
        Check( IsPixelRed( Graphic, 92, 418 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    { multipage load from file stream opened exclusive }
    Graphic := TTiffGraphic.Create;
    Stream  := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'race.tif'), fmOpenRead or fmShareExclusive );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 741 );
        Check( Graphic.Height = 570 );
        Check( IsPixelLight( Graphic, 186, 227 ) );
        Check( IsPixelLight( Graphic, 726, 539 ) );
        Check( IsPixelDark( Graphic, 553, 25 ) );
        Check( IsPixelDark( Graphic, 102, 537 ) );
        Check( IsPixelRed( Graphic, 92, 418 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;

    { multipage load from file stream opened shared }
    Graphic := TTiffGraphic.Create;
    Stream  := nil;
    try
        Stream := TFileStream.Create( TestImageFileName( 'race.tif'), fmOpenRead or fmShareDenyWrite );
        Stream.Seek( 0, soFromBeginning );
        Check( Graphic.GetImageCount( Stream ) = 2 );

        Stream.Seek( 0, soFromBeginning );
        Graphic.ImageToLoad := 2;
        Graphic.LoadFromStream( Stream );
        Check( Graphic.ImageFormat = ifTrueColor );
        Check( Graphic.Width = 741 );
        Check( Graphic.Height = 570 );
        Check( IsPixelLight( Graphic, 186, 227 ) );
        Check( IsPixelLight( Graphic, 726, 539 ) );
        Check( IsPixelDark( Graphic, 553, 25 ) );
        Check( IsPixelDark( Graphic, 102, 537 ) );
        Check( IsPixelRed( Graphic, 92, 418 ) );
    finally
        Stream.Free;
        Graphic.Free;
    end;


    { With free image prior to 3.16 the second page of this tiff would raise an access violation }
    if FileExists(PrivateTestImageFileName( 'C1892302-F165-422E-B0B9-A7F93AD03E0C.tif' )) then
    begin
        Graphic := TTiffGraphic.Create;
        try
            Check( Graphic.GetImageCountFromFile( PrivateTestImageFileName('C1892302-F165-422E-B0B9-A7F93AD03E0C.tif') ) = 3 );

            Graphic.ImageToLoad := 2;
            Graphic.LoadFromFile( PrivateTestImageFileName('C1892302-F165-422E-B0B9-A7F93AD03E0C.tif') );
            Check( Graphic.ImageFormat = ifBlackWhite );
            Check( Graphic.Width = 2552 );
            Check( Graphic.Height = 3510 );
            Check( IsPixelLight( Graphic, 1090, 214 ) );
            Check( IsPixelLight( Graphic, 1263, 3138 ) );
            Check( IsPixelDark( Graphic, 411, 3216 ) );
            Check( IsPixelDark( Graphic, 1841, 1311 ) );
        finally
            Graphic.Free;
        end;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestWbmp;
var
    Graphic     : TWbmpGraphic;
begin
    { fax g3 does not have signature so need to use TWbmpGraphic explicitly }
    Graphic := TWbmpGraphic.Create;
    try
        Graphic.LoadFromFile( TestImageFileName( 'feet.wbmp' ) );

        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 28 );
        Check( Graphic.Height = 32 );
        Check( IsPixelLight( Graphic, 9, 5 ) );
        Check( IsPixelLight( Graphic, 19, 5 ) );
        Check( IsPixelDark( Graphic, 11, 1 ) );
        Check( IsPixelDark( Graphic, 20, 30 ) );
    finally
        Graphic.Free;
    end;
end;


{--------------------------------------------------------------------------}

procedure TImageReadTest.TestWmf;
var
    Graphic     : TMetaFileGraphic;
begin

    Graphic := TMetaFileGraphic.Create;
    try
        Graphic.BackgroundColor := MakeRGB( 0, 255, 0 );
        Graphic.LoadFromFile( TestImageFileName( 'Try our pizzas.wmf' ) );
        Check( Graphic.ImageFormat = ifTrueColor );

        Check( Graphic.Height = Round(Screen.PixelsPerInch * 1.36666666) );
        Check( Graphic.Width  = Round(Screen.PixelsPerInch * 1.36666666 * 2.84146341463) ) ;

        Check( IsPixelGreen( Graphic, 10, 10 ) );

    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestXbm;
var
    Graphic     : TDibGraphic;
begin
    Graphic := CreateGraphicFromFile( TestImageFileName( 'pumpkin.xbm' ), TXbmGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifBlackWhite );
        Check( Graphic.Width = 206 );
        Check( Graphic.Height = 187 );
        Check( IsPixelLight( Graphic, 101, 178 ) );
        Check( IsPixelLight( Graphic, 96, 79 ) );
        Check( IsPixelDark( Graphic, 70, 87 ) );
        Check( IsPixelDark( Graphic, 120, 140 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageReadTest.TestXpm;
var
    Graphic     : TDibGraphic;
begin
    Graphic := CreateGraphicFromFile( TestImageFileName( 'Irbis.xpm' ), TXpmGraphic, 1 );
    try
        Check( Graphic.ImageFormat = ifColor256 );
        Check( Graphic.Width = 64 );
        Check( Graphic.Height = 71 );
        Check( IsPixelLight( Graphic, 3, 9 ) );
        Check( IsPixelLight( Graphic, 60, 18 ) );
        Check( IsPixelDark( Graphic, 13, 34 ) );
        Check( IsPixelDark( Graphic, 36, 32 ) );
    finally
        Graphic.Free;
    end;
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestBmp;
var
    Source       : TJpegGraphic;
    Dest         : TBitmapGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TJpegGraphic.Create;
        Dest   := TBitmapGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.jpg') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('bmp');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;

        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );



    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestDcx;
var
    Source       : TBitmapGraphic;
    Dest         : TDcxGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TDcxGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );

        TempFileName := TempTestImageFileName('dcx');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        { 4.03 PCX always saved in true color }


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        { saved as color 256 }
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );


    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;
{--------------------------------------------------------------------------}

function TImageWriteTest.GetPage( const ImageNo     : Integer;
                                  var   FreeGraphic : Boolean ) : TGraphic;
begin
    Result      := FPageToSave;
    FreeGraphic := False;
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestDcxMultiPage;
var
    Source       : TBitmapGraphic;
    Dest         : TDcxGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Check( TDcxGraphic.CanSave );

    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TDcxGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );

        TempFileName := TempTestImageFileName('dcx');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        { 4.03 PCX always saved in true color }


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 2;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 3;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 4;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 5;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        { saved as gray 256 }
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 6;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Check( Dest.GetImageCountFromFile(TempFileName) = 6 );

        FPageToSave := Source;
        Dest.MultiSaveToFile( TempFileName, 3, GetPage );
        Check( Dest.GetImageCountFromFile(TempFileName) = 3 );


    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestDds;
begin
    Check( not TDdsGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestEps;
begin
    Check( not TEpsGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestFaxG3;
begin
    Check( not TFaxG3Graphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestIco;
var
    Source       : TPngGraphic;
    Dest         : TIconGraphic;
    TempFileName : String;
    Color        : TRgb;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TPngGraphic.Create;
        Dest   := TIconGraphic.Create;

        Source.OverwriteFileBackground := True;
        Color.Red     := 0;
        Color.Green   := 255;
        Color.Blue    := 0;
        Source.BackgroundColor := Color;
        Source.LoadFromFile( TestImageFileName( 'pngwithbk.png') );
        Check( Source.Width = 32 );
        Check( Source.Height = 32 );

        TempFileName := TempTestImageFileName('ico');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;


        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );

        { icon size may be sensitive to windows display "scaling", so
          Dest.Width may not be 32 pixels }
        Check( Dest.Width > 10 );
        Check( Dest.Width < 200 );
        Check( Dest.Width = Dest.Height );

        if Dest.Width = 32 then
        begin
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelGreen( Dest, 1, 6 ) );
            Check( IsPixelGreen( Dest, 1, 28 ) );

            Dest.Assign(Source);
            Convert( Dest, ifColor256 );
            Dest.SaveToFile( TempFileName );
            Dest.Clear;
            Check( Dest.Width = 0 );
            Dest.LoadFromFile( TempFileName );
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            { saved as True color }
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelGreen( Dest, 1, 6 ) );
            Check( IsPixelGreen( Dest, 1, 28 ) );


            Dest.Assign(Source);
            Convert( Dest, ifColor16 );
            Dest.SaveToFile( TempFileName );
            Dest.Clear;
            Check( Dest.Width = 0 );
            Dest.LoadFromFile( TempFileName );
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            { saved as True color }
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelGreen( Dest, 1, 6 ) );
            Check( IsPixelGreen( Dest, 1, 28 ) );

            Dest.Assign(Source);
            Convert( Dest, ifGray256 );
            Dest.SaveToFile( TempFileName );
            Dest.Clear;
            Check( Dest.Width = 0 );
            Dest.LoadFromFile( TempFileName );
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            { saved as True color }
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelDark( Dest, 1, 6 ) );
            Check( IsPixelDark( Dest, 1, 28 ) );

            Dest.Assign(Source);
            Convert( Dest, ifGray16 );
            Dest.SaveToFile( TempFileName );
            Dest.Clear;
            Check( Dest.Width = 0 );
            Dest.LoadFromFile( TempFileName );
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            { saved as True color }
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelDark( Dest, 1, 6 ) );
            Check( IsPixelDark( Dest, 1, 28 ) );

            Dest.Assign(Source);
            Convert( Dest, ifBlackWhite );
            Dest.SaveToFile( TempFileName );
            Dest.Clear;
            Check( Dest.Width = 0 );
            Dest.LoadFromFile( TempFileName );
            Check( Dest.Width = Source.Width );
            Check( Dest.Height = Source.Height );
            { saved as True color }
            Check( Dest.ImageFormat = ifTrueColor );
            Check( not IsPixelLight( Dest, 1, 6 ) );
            Check( not IsPixelLight( Dest, 1, 28 ) );
            Check( IsPixelDark( Dest, 1, 6 ) );
            Check( IsPixelDark( Dest, 1, 28 ) );
        end;



    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestJ2k;
var
    Source       : TJpegGraphic;
    Dest         : TJ2kGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TJpegGraphic.Create;
        Dest   := TJ2kGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.jpg') );
        TempFileName := TempTestImageFileName('j2k');

        { first test with high compression, low save quality }
        Dest.Assign(Source);
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 20;
            Dest.SaveToStream( FileStream );
            Check( FileStream.Size < 60 * 1024 );
        finally
            FileStream.Free;
        end;


        Dest.Assign(Source);
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 80;
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width  = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );
    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestJng;
begin
    Check( not TJngGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestJp2;
var
    Source       : TJpegGraphic;
    Dest         : TJp2Graphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TJpegGraphic.Create;
        Dest   := TJp2Graphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.jpg') );
        TempFileName := TempTestImageFileName('jp2');

        { first test with high compression, low save quality }
        Dest.Assign(Source);
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 20;
            Dest.SaveToStream( FileStream );
            Check( FileStream.Size < 60 * 1024 );
        finally
            FileStream.Free;
        end;

        Dest.Assign(Source);
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 80;
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width  = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );
    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestJpg;
var
    Source       : TJpegGraphic;
    Dest         : TJpegGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TJpegGraphic.Create;
        Dest   := TJpegGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.jpg') );
        TempFileName := TempTestImageFileName('jpg');

        Dest.Assign(Source);
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 95;
            Dest.SaveToStream( FileStream );
            Check( FileStream.Size > 59000 );
        finally
            FileStream.Free;
        end;

        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveQuality := 15;
            Dest.SaveToStream( FileStream );
            Check( FileStream.Size < 15000 );
        finally
            FileStream.Free;
        end;

        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width  = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifGray256 );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );
    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;
{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestMng;
begin
    Check( not TMngGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPict;
begin
    Check( not TPictGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPcx;
var
    Source       : TBitmapGraphic;
    Dest         : TPcxGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TPcxGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('pcx');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        { 4.03 PCX always saved in true color }

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPdf;
var
    Source       : TBitmapGraphic;
    Dest         : TPdfGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TPdfGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('pdf');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        { size is different becauase of PDF rendering dpi.

          So strange, there is a 1 pixel difference in rendering between 32 and 64 with the
          PDFIum library!! }
        Check( Dest.Width = 963 );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = 963 );
        Check( Dest.Height =709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ColorConversion := EnPdfium.pdfGray;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = 963 );
        Check( Dest.Height =709 );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ColorConversion := EnPdfium.pdfColor;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height =709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ColorConversion := EnPdfium.pdfGray;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = 963 );
        Check( Dest.Height =709 );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ColorConversion := EnPdfium.pdfBW;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = 963 );
        Check( Dest.Height =709 );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPdfMultiPage;
var
    Source       : TBitmapGraphic;
    Dest         : TPdfGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Check( TDcxGraphic.CanSave );

    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TPdfGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );

        TempFileName := TempTestImageFileName('pdf');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        { size is different becauase of PDF rendering dpi.

          So strange, there is a 1 pixel difference in rendering between 32 and 64 with the
          PDFIum library!! }
        Check( Dest.Width = 963 );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 2;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 3;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 4;
        Dest.ColorConversion := EnPdfium.pdfGray;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 5;
        Dest.ColorConversion := EnPdfium.pdfGray;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 6;
        Dest.ColorConversion := EnPdfium.pdfBW;
        Dest.LoadFromFile( TempFileName );
        Check( (Dest.Width = 962) or (Dest.Width = 963) );
        Check( Dest.Height = 709 );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 14, 14 ) );
        Check( IsPixelLight( Dest, 929, 675 ) );
        Check( IsPixelDark( Dest, 95, 564 ) );
        Check( IsPixelDark( Dest, 729, 389 ) );

        Check( Dest.GetImageCountFromFile(TempFileName) = 6 );

        FPageToSave := Source;
        Dest.MultiSaveToFile( TempFileName, 3, GetPage );
        Check( Dest.GetImageCountFromFile(TempFileName) = 3 );


    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;


{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPng;
var
    Source           : TBitmapGraphic;
    Dest             : TPngGraphic;
    TempFileName     : String;
    FileStream       : TFileStream;
    LowCompressSize  : Int64;
    HighCompressSize : Int64;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TPngGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('png');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );

        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            { test saving transparent true color image }
            {$ifdef __Delphi2009AndAbove}
            Dest.SaveTransparent      := True;
            Dest.TransparentSaveColor := Dest.RGB[0,0];
            {$endif}

            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;

        Dest.Clear;
        Check( Dest.Width = 0 );

        Dest.OverwriteFilebackground := True;
        Dest.BackgroundColor         := MakeRGB( 0, 255, 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );

        {$ifdef __Delphi2009AndAbove}
        Check( IsPixelGreen( Dest, 23, 37 ) );
        {$endif}

        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        {$ifdef __Delphi2009AndAbove}
        Dest.SaveTransparent      := True;
        Dest.TransparentSaveColor := Dest.RGB[0,0];
        {$endif}
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.OverwriteFilebackground := True;
        Dest.BackgroundColor         := MakeRGB( 0, 255, 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        {$ifdef __Delphi2009AndAbove}
        { transparent images are loaded back in true color }
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelGreen( Dest, 23, 37 ) );
        {$else}
        Check( Dest.ImageFormat = ifColor256 );
        {$endif}
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        { all other tests are not transparent }
        {$ifdef __Delphi2009AndAbove}
        Dest.SaveTransparent      := False;
        {$endif}

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        { test different compression level }
        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.CompressionLevel := pcLow;
        Dest.SaveToFile( TempFileName );
        LowCompressSize := EnMisc.GetFileSize(TempFileName);
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.CompressionLevel := pcHigh;
        Dest.SaveToFile( TempFileName );
        HighCompressSize := EnMisc.GetFileSize(TempFileName);

        { this will validate that different compression levels are working }
        Check( HighCompressSize + 10000 < LowCompressSize );

        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPpm;
var
    Source       : TPpmGraphic;
    Dest         : TPpmGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TPpmGraphic.Create;
        Dest   := TPpmGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.ppm') );
        Check( Source.Width = 257 );
        Check( Source.Height = 189 );

        TempFileName := TempTestImageFileName('ppm');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.Mode  := pmAscii;
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.Mode  := pmAscii;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.Mode  := pmAscii;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.Mode  := pmAscii;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.Mode  := pmAscii;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.Mode  := pmAscii;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );


        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.Mode  := pmBinary;
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.Mode  := pmBinary;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.Mode  := pmBinary;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.Mode  := pmBinary;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.Mode  := pmBinary;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.Mode  := pmBinary;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 18, 10 ) );
        Check( IsPixelLight( Dest, 180, 36 ) );
        Check( IsPixelDark( Dest, 24, 150 ) );
        Check( IsPixelDark( Dest, 187, 51 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;


{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestPsd;
begin
    Check( not TPsdGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestTga;
var
    Source       : TBitmapGraphic;
    Dest         : TTgaGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TTgaGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('tga');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestTiff;
var
    Source       : TBitmapGraphic;
    Dest         : TTiffGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
    Raised       : Boolean;

    HeaderStream : TFileStream;


begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TTiffGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('tif');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Raised := False;
            try
                Dest.Compression := tcGroup4;
            except
                Raised := True;
            end;
            Check( Raised ); { compression not supported with image format }

            Dest.Compression := tcNone;
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Dest.Compression := tcPackbits;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcNone );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );

        Raised := False;
        try
            Dest.Compression := tcGroup3_1d;
        except
            Raised := True;
        end;
        Check( Raised ); { compression not supported with image format }

        Dest.Compression := tcLZW;

        Dest.DocumentNameTag       := 'DocumentNameTag';
        Dest.ImageDescriptionTag   := 'ImageDescriptionTag';
        Dest.ScannerMakeTag        := 'ScannerMakeTag';
        Dest.ScannerModelTag       := 'ScannerModelTag';
        Dest.PageNameTag           := 'PageNameTag';
        Dest.SoftwareTag           := 'SoftwareTag';
        Dest.DateTimeTag           := 'DateTimeTag';
        Dest.ArtistTag             := 'ArtistTag';
        Dest.HostComputerTag       := 'HostComputerTag';
        Dest.SaveToFile( TempFileName );

        Dest.Clear;
        Dest.DocumentNameTag       := '';
        Dest.ImageDescriptionTag   := '';
        Dest.ScannerMakeTag        := '';
        Dest.ScannerModelTag       := '';
        Dest.PageNameTag           := '';
        Dest.SoftwareTag           := '';
        Dest.DateTimeTag           := '';
        Dest.ArtistTag             := '';
        Dest.HostComputerTag       := '';
        Dest.Compression := tcPackbits;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcLZW );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

(* tiff tag saving does not work in FreeImage at the moment this was
   implemented. See post,

http://sourceforge.net/p/freeimage/discussion/36110/thread/6812a437/

        Check( Dest.DocumentNameTag       = 'DocumentNameTag' );
        Check( Dest.ImageDescriptionTag   = 'ImageDescriptionTag' );
        Check( Dest.ScannerMakeTag        = 'ScannerMakeTag' );
        Check( Dest.ScannerModelTag       = 'ScannerModelTag' );
        Check( Dest.PageNameTag           = 'PageNameTag' );
        Check( Dest.SoftwareTag           = 'SoftwareTag' );
        Check( Dest.DateTimeTag           = 'DateTimeTag' );
        Check( Dest.ArtistTag             = 'ArtistTag' );
        Check( Dest.HostComputerTag       = 'HostComputerTag' );
*)

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );

        Raised := False;
        try
            Dest.Compression := tcGroup3_2d;
        except
            Raised := True;
        end;
        Check( Raised ); { compression not supported with image format }

        Dest.Compression := tcPackbits;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Dest.Compression := tcNone;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcPackbits );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );

        Raised := False;
        try
            Dest.Compression := tcJPEG;
        except
            Raised := True;
        end;
        Check( Raised ); { compression not supported with image format }

        Dest.Compression := tcZLib;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Dest.Compression := tcNone;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcZLib );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Raised := False;
        try
            Dest.Compression := tcGroup3_2d;
        except
            Raised := True;
        end;
        Check( Raised ); { compression not supported with image format }
        Dest.Compression := tcJPEG;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Dest.Compression := tcNone;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcJPEG );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Raised := False;
        try
            Dest.Compression := tcJPEG;
        except
            Raised := True;
        end;
        Check( Raised ); { compression not supported with image format }
        Dest.Compression := tcGroup4;
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Dest.Compression := tcNone;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Compression = tcGroup4 );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

    { test saving to stream, but not at beginning of stream }
    Source       := nil;
    Dest         := nil;
    HeaderStream := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TTiffGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        HeaderStream := TFileStream.Create( TestImageFileName( 'tigers.bmp'), fmOpenRead );

        TempFileName := TempTestImageFileName('tif');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.Compression := tcPackbits;

            { copy bytes at the beginning of the stream }
            FileStream.CopyFrom( HeaderStream, 0 );

            Dest.SaveToStream( FileStream );

            FileStream.Seek( HeaderStream.Size, soFromBeginning );

            { test loading from stream not at beginning }
            Dest.Clear;
            Dest.LoadFromStream( FileStream );

        finally
            FileStream.Free;
        end;

        Check( Dest.Compression = tcPackbits );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        HeaderStream.Free;
        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestTiffMultiPage;
var
    Source       : TBitmapGraphic;
    Dest         : TTiffGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Check( TDcxGraphic.CanSave );

    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TTiffGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );

        TempFileName := TempTestImageFileName('tif');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate or fmShareDenyWrite);
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 2;
        Check( Dest.GetImageCount(TempFileName) = 2 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 3;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 4;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.AppendToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 5;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray16 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );

        FileStream := TFileStream.Create( TempFileName, fmOpenReadWrite or fmShareExclusive);
        try
            Dest.AppendToStream( FileStream );
        finally
            FileStream.Free;
        end;

        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.ImageToLoad := 6;
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        FileStream := TFileStream.Create( TempFileName, fmOpenReadWrite );
        try
            Check( Dest.GetImageCount(FileStream) = 6 );
        finally
            FileStream.Free;
        end;

        Check( Dest.GetImageCountFromFile(TempFileName) = 6 );

        FPageToSave := Source;
        Dest.MultiSaveToFile( TempFileName, 3, GetPage );
        Check( Dest.GetImageCountFromFile(TempFileName) = 3 );

        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            FPageToSave := Source;
            Dest.MultiSaveToStream( FileStream, 3, GetPage );
            Check( Dest.GetImageCount(FileStream) = 3 );
        finally
            FileStream.Free;
        end;

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestWbmp;
var
    Source       : TBitmapGraphic;
    Dest         : TWbmpGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TWbmpGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('wbmp');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        { moved slightly from other tests because of dithering is slightly different.
          Visually verified image and it is fine }
        Check( IsPixelLight( Dest, 620, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifBlackWhite );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestWmf;
var
    Source       : TBitmapGraphic;
    Dest         : TMetaFileGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TMetaFileGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('wmf');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestXbm;
begin
    Check( not TXbmGraphic.CanSave );
end;

{--------------------------------------------------------------------------}

procedure TImageWriteTest.TestXpm;
var
    Source       : TBitmapGraphic;
    Dest         : TXpmGraphic;
    TempFileName : String;
    FileStream   : TFileStream;
begin
    Source       := nil;
    Dest         := nil;
    TempFileName := '';
    try
        Source := TBitmapGraphic.Create;
        Dest   := TXpmGraphic.Create;

        Source.LoadFromFile( TestImageFileName( 'tigers.bmp') );
        Check( Source.Width = 641 );
        Check( Source.Height = 472 );

        TempFileName := TempTestImageFileName('xpm');

        Dest.Assign(Source);
        Convert( Dest, ifTrueColor );
        FileStream := TFileStream.Create( TempFileName, fmCreate );
        try
            Dest.SaveToStream( FileStream );
        finally
            FileStream.Free;
        end;
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifTrueColor );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifColor16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        { XPM save is smart so if less than 256 colors, will be saved
          as palettized even if the source was converted to true color
          before saving }
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifGray16 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifColor256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifColor256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );


        Dest.Assign(Source);
        Convert( Dest, ifGray256 );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

        Dest.Assign(Source);
        Convert( Dest, ifBlackWhite );
        Dest.SaveToFile( TempFileName );
        Dest.Clear;
        Check( Dest.Width = 0 );
        Dest.LoadFromFile( TempFileName );
        Check( Dest.Width = Source.Width );
        Check( Dest.Height = Source.Height );
        Check( Dest.ImageFormat = ifGray256 );
        Check( IsPixelLight( Dest, 9, 9 ) );
        Check( IsPixelLight( Dest, 619, 450 ) );
        Check( IsPixelDark( Dest, 63, 376 ) );
        Check( IsPixelDark( Dest, 489, 259 ) );

    finally
        if TempFileName <> '' then
            SysUtils.DeleteFile(TempFileName);

        Source.Free;
        Dest.Free;
    end;

end;

{--------------------------------------------------------------------------}

procedure TEnvisionAssistedTest.ReceiveImageFromScanner(
                                       const DibHandle    : THandle;
                                       const XDpi         : Word;
                                       const YDpi         : Word;
                                       const CallBackData : Pointer );
var
    Graphic : TDibGraphic;
begin
    Inc(FPagesScanned);

    Graphic := TDibGraphic.Create;
    try
        Graphic.AssignFromDIBHandle(DibHandle);

        Check( Graphic.Width = 1700 );
        Check( Graphic.Height = 2200 );

        //Check( IsPixelDark( Graphic, 591, 935 ));
        ////Check( IsPixelDark( Graphic, 986, 953 ));
        //Check( IsPixelDark( Graphic, 80, 2 ));
        //Check( IsPixelDark( Graphic, 1645, 2171 ));

        ////Check( IsPixelLight( Graphic, 929, 1387 ));
        //Check( IsPixelLight( Graphic, 839, 70 ));

    finally
        Graphic.Free;
    end;


end;

{--------------------------------------------------------------------------}

procedure TEnvisionAssistedTest.TestTwainScanning;
var
    ScannerIndex : Integer;
begin
    Check( Scanner.IsConfigured );

    ScannerIndex := Scanner.Scanners.IndexOf('TWAIN2 Software Scanner');

    if ScannerIndex < 0 then
    begin
        MessageDlg( 'TWAIN test database source not installed.'#13 +
                    'See instructions in the file .\Twain\TestDataSource\Readme.txt',
                    mtInformation, [mbOk], 0);

        Check(False);
    end;

    FPagesScanned := 0;

    Scanner.ScannerIndex := ScannerIndex;
    Scanner.Acquire( ReceiveImageFromScanner, nil);

    MessageDlg( IntToStr(FPagesScanned) + ' pages scanned', mtInformation, [mbOk], 0);

end;

{--------------------------------------------------------------------------}


initialization

    { register formats not registered by default. See Envision.Inc }
    RegisterDibGraphic('BMP', 'Bitmap file', TBitmapGraphic);
    RegisterDibGraphic('ICO', 'Windows icon', TIconGraphic);
    RegisterDibGraphic('WMF', 'Windows meta file', TMetaFileGraphic);
    RegisterDibGraphic('EMF', 'Enhanced meta file', TMetaFileGraphic);

    TestFramework.RegisterTest(TEnvisionBaseTest.Suite);
    TestFramework.RegisterTest(TImageReadTest.Suite);
    TestFramework.RegisterTest(TImageWriteTest.Suite);
    TestFramework.RegisterTest(TEnvisionAssistedTest.Suite);

end.
