{----------------------------------------------------------------------------
|
| Library: Envision
|
| Module: EnPdfium
|
| Description: Helper unit to interfade with PDFium library.
|
|              See the __EnablePDFium conditional in Envision.Inc to
|              enable or disable this support
|
|
| History: Oct 25 2018, Michel Brazeau
|
|---------------------------------------------------------------------------}
unit EnPdfium;

{$I Envision.Inc}

interface

{$B-,X+}

{$ifndef __EnablePDFium}
!! You must enabled  __EnablePDFium in Envision.Inc to use this unit !!
{$endif}

uses
    Classes,          { for TStream }
    PdfiumCore,       { for TPdfDocument }
    EnDiGrph;         { for TDibGraphic }

type

TPdfColorConversion = (pdfBW, pdfGray, pdfColor);

type

TPdfiumReader = class(TObject)
protected
    FPdfDocument     : TPdfDocument;
    FLoaded          : Boolean;


public

    constructor Create;
    destructor Destroy; override;


    { PDF file or stream must be loaded before calling other methods }
    procedure LoadFromFile( const FileName : String );
    procedure LoadFromStream( const Stream   : TStream;
                              const Password : String );

    function PageCount : Integer;

   {  Returned graphic must be released by user.

      PageNo >= 1

      Best quality for DPI is multiples of 72, for example, 216
   }
   function GetPageAsImage( const PageNo          : Integer;
                            const DPI             : Integer;
                            const ColorConversion : TPdfColorConversion ) : TDibGraphic;

   function GetPageAsImageClass( const ImageClass: TDibGraphicClass;
                            const PageNo          : Integer;
                            const DPI             : Integer;
                            const ColorConversion : TPdfColorConversion ) : TDibGraphic;

   { Get the text of a page, PageNo >= 1 }
   function PageText( const PageNo : Integer ) : String;

   function GetAuthor : String;
   function GetTitle : String;
   function GetKeywords : String;
   function GetSubject : String;
   function GetProducer : String;
   function GetCreator : String;

end;


(************************************************************************

  * *   * ***  *    **** *   * **** *   * *****   *   ***** *  **  *   *
  * ** ** *  * *    *    ** ** *    **  *   *    * *    *   * *  * **  *
  * * * * ***  *    **   * * * **   * * *   *   *   *   *   * *  * * * *
  * *   * *    *    *    *   * *    *  **   *   *****   *   * *  * *  **
  * *   * *    **** **** *   * **** *   *   *   *   *   *   *  **  *   *

 ************************************************************************)

implementation

uses
    {$ifdef __DelphiD13AndAbove}Winapi.Windows,{$else}Windows,{$endif}

    Math, { for Math.SetExceptionMask }
    EnTransf,      { for TImageFormatTransform }

    PdfiumAPI,     { for PDFiumDone }

    SysUtils,      { for Exception, DeleteFile  }
    EnMisc;        { for ifTrueColor }

{---------------------------------------------------------------------------}

constructor TPdfiumReader.Create;
begin
    inherited Create;

    FLoaded          := False;
    FPdfDocument     := TPdfDocument.Create;

end;

{---------------------------------------------------------------------------}

destructor TPdfiumReader.Destroy;
begin
    FPdfDocument.Free;

    inherited Destroy;
end;

{---------------------------------------------------------------------------}

procedure TPdfiumReader.LoadFromFile( const FileName : String );
begin
    try
        FPdfDocument.LoadFromFile( FileName );
        FLoaded := True;
    except
        FLoaded := False;
        raise;
    end;
end;

{---------------------------------------------------------------------------}

procedure TPdfiumReader.LoadFromStream( const Stream   : TStream;
                                        const Password : String  );
begin
    try
        Stream.Seek( 0, soFromBeginning );
        {$WARNINGS OFF}
        FPdfDocument.LoadFromStream( Stream, Password );
        {$WARNINGS ON}
        FLoaded := True;
    except
        FLoaded := False;
        raise;
    end;
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.PageCount : Integer;
begin
    if not FLoaded then
        raise Exception.Create ('PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.PageCount;
end;

{---------------------------------------------------------------------------}

function GetPage( const PdfDocument : TPdfDocument;
                           PageNo      : Integer
                   ) : TPdfPage;


    {$ifdef CPUX64}
    var
    SavedMask : TArithmeticExceptionMask;
    {$endif}

begin

    if PageNo > PdfDocument.PageCount then
        PageNo := PdfDocument.PageCount;

    if PageNo < 1 then
        PageNo := 1;

    { MB Oct 7, 2024, prevent occasional floating point exception in some Delphi version (10,11) }
    {$ifdef CPUX64}
    SavedMask := Math.GetExceptionMask;
    {$endif}
    try

		{ 4.06 Preserve SavedMask }
        {$ifdef CPUX64}
        Math.SetExceptionMask(SavedMask + [exInvalidOp, exPrecision, exDenormalized]);
        {$endif}

        { zero based }
        Result   := PdfDocument.Pages[PageNo-1];

    finally
        {$ifdef CPUX64}
        Math.SetExceptionMask(SavedMask);
        {$endif}
    end;
end;

function TPdfiumReader.GetPageAsImage( const PageNo          : Integer;
                                       const DPI             : Integer;
                                       const ColorConversion : TPdfColorConversion ) : TDibGraphic;
var
    ImageFormat   : EnMisc.TImageFormat;
    Transform     : TImageFormatTransform;

    Width         : Integer;
    Height        : Integer;

    Page          : TPdfPage;

    { 4.05 set mask locally instead of unit initialization }
    {$ifdef CPUX64}
    SavedMask : TArithmeticExceptionMask;
    {$endif}



begin

    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Transform := nil;
    try


        Page   := GetPage( FPdfDocument, PageNo );

        Result := TDibGraphic.Create;
        try

            { 4.03 }
            Width  := EnMisc.Ceiling(Page.Width * (DPI/72));
            Height := EnMisc.Ceiling(Page.Height * (DPI/72));

            case ColorConversion of
                pdfBW   : ImageFormat := ifBlackWhite;
                pdfGray : ImageFormat := ifGray256;
                else
                    ImageFormat := ifTrueColor;
            end;

            Result.NewImage( Width, Height,
                             ImageFormat,
                             nil, DPI, DPI );

            Result.Fill( EnDiGrph.CWhiteRgb );


            {$ifdef CPUX64}
            SavedMask := Math.GetExceptionMask;
            {$endif}
            try

	        	{ 4.06 Preserve SavedMask }
                {$ifdef CPUX64}
                Math.SetExceptionMask(SavedMask + [exInvalidOp, exPrecision, exDenormalized]);
                {$endif}


                Page.Draw(Result.DC, 0, 0, Width, Height, prNormal, [proAnnotations, proPrinting, proNoCatch]);

            finally
                {$ifdef CPUX64}
                Math.SetExceptionMask(SavedMask);
                {$endif}
            end;


            { Pdfium library already rotates the image. No need to apply rotation
               manually
            }

        except
            Result.Free;
            Raise;
        end;

    finally
        FreeAndNil( Transform );
    end;
end;

function TPdfiumReader.GetPageAsImageClass(const ImageClass: TDibGraphicClass;
                                       const PageNo          : Integer;
                                       const DPI             : Integer;
                                       const ColorConversion : TPdfColorConversion ) : TDibGraphic;
var
    ImageFormat   : EnMisc.TImageFormat;
    Transform     : TImageFormatTransform;

    Width         : Integer;
    Height        : Integer;

    Page          : TPdfPage;

    { 4.05 set mask locally instead of unit initialization }
    {$ifdef CPUX64}
    SavedMask : TArithmeticExceptionMask;
    {$endif}



begin

    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Transform := nil;
    try


        Page   := GetPage( FPdfDocument, PageNo );

        Result := ImageClass.Create;
        try

            { 4.03 }
            Width  := EnMisc.Ceiling(Page.Width * (DPI/72));
            Height := EnMisc.Ceiling(Page.Height * (DPI/72));

            case ColorConversion of
                pdfBW   : ImageFormat := ifBlackWhite;
                pdfGray : ImageFormat := ifGray256;
                else
                    ImageFormat := ifTrueColor;
            end;

            Result.NewImage( Width, Height,
                             ImageFormat,
                             nil, DPI, DPI );

            Result.Fill( EnDiGrph.CWhiteRgb );

            {$ifdef CPUX64}
            SavedMask := Math.GetExceptionMask;
            {$endif}
            try

	        	{ 4.06 Preserve SavedMask }
                {$ifdef CPUX64}
                Math.SetExceptionMask(SavedMask + [exInvalidOp, exPrecision, exDenormalized]);
                {$endif}

                Page.Draw(Result.DC, 0, 0, Width, Height, prNormal, [proAnnotations, proPrinting, proNoCatch]);

            finally
                {$ifdef CPUX64}
                Math.SetExceptionMask(SavedMask);
                {$endif}
            end;

            { Pdfium library already rotates the image. No need to apply rotation
               manually
            }

        except
            Result.Free;
            Raise;
        end;

    finally
        FreeAndNil( Transform );
    end;
end;


{---------------------------------------------------------------------------}

function TPdfiumReader.PageText( const PageNo : Integer ) : String;
var
    Page          : TPdfPage;

begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Page   := GetPage( FPdfDocument, PageNo );

    Result := Page.ReadText( 0, Page.GetCharCount );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetAuthor : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Author' );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetTitle : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Title' );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetSubject : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Subject' );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetKeywords : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Keywords' );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetCreator : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Creator' );
end;

{---------------------------------------------------------------------------}

function TPdfiumReader.GetProducer : String;
begin
    RaiseIfNot( FLoaded, 'PDF not loaded in TPdfiumReader');

    Result := FPdfDocument.GetMetaText( 'Producer' );
end;

{---------------------------------------------------------------------------}

var

TempPDFiumFolder      : String;
FullPdfiumDLLFileName : String;


{$ifdef __EmbedDLL}

const

CPdfiumDLLResourceName = 'PDFiumDLL';

{$ifdef Win32}

{$R EnPdfium32.RES}

{$else}

{$R EnPdfium64.RES}

{$endif}


procedure ExtractPdfiumDLLs;
var
    ResStream   : TResourceStream;

begin
    if FileExists(AddBackSlashToPath( ExtractFilePath(ParamStr(0)) ) + CPdfiumDllName) then
        Exit;

    ResStream := nil;
    try
        ResStream := TResourceStream.Create(HInstance, CPdfiumDLLResourceName, RT_RCDATA);

        TempPDFiumFolder := AddBackSlashToPath(EnMisc.CreateTempFolder( 'Env' ));

        FullPdfiumDLLFileName := TempPDFiumFolder + CPdfiumDllName;

        ResStream.SaveToFile(FullPdfiumDLLFileName);

        PdfiumCore.PDFiumDllDir := TempPDFiumFolder;

    finally
        ResStream.Free;
    end;
end;

{$endif}

procedure UnsupportedHandler(pThis: PUNSUPPORT_INFO; nType: Integer); cdecl;
begin
end;

var

UnsupportInfo: TUnsupportInfo = (
  version: 1;
  FSDK_UnSupport_Handler: UnsupportedHandler;
);

PdfDocument : PdfiumCore.TPdfDocument;

{---------------------------------------------------------------------------}

{$ifdef CPUX64}
var
    { 4.05 set mask locally }

    SavedMask : TArithmeticExceptionMask;
{$endif}

initialization



    TempPDFiumFolder      := '';
    FullPdfiumDLLFileName := '';

    {$ifdef __EmbedDLL}
    ExtractPdfiumDLLs;
    {$endif}

    { this is important in 64 bits, otherwise

      Page.Draw(Result.DC, 0, 0, Width, Height, prNormal, [proAnnotations, proPrinting, proNoCatch]);

      will raise invalid operation. Also need to add exPrecision otherwise another exception
      is raised on startup .

      Also exDenormalized is required to prevent exception with .\TestImagesPrivate\TigersMathException.tif.
      4.05 exception no longer occurs addted to automated tests

      See https://stackoverflow.com/questions/9359223/delphi-xe2-twebbrowser-float-divide-by-zero
          https://stackoverflow.com/questions/34777301/disable-floating-point-exceptions-during-a-single-function
    }


    {$ifdef CPUX64}
    SavedMask := Math.GetExceptionMask;
    {$endif}
    try

      	{ 4.06 Preserve SavedMask }
        {$ifdef CPUX64}
        Math.SetExceptionMask(SavedMask + [exInvalidOp, exPrecision, exDenormalized]);
        {$endif}

        { replace the default FSDK_UnSupport_Handler which raises exception.
          this prevented loading for example a PDF with embedded form }

        { create a TPdfDocument to force loading the DLL }
        PdfDocument := TPdfDocument.Create;
        PdfDocument.Free;

        PdfSetUnSpObjProcessHandler(@UnsupportInfo);
    finally
        {$ifdef CPUX64}
        Math.SetExceptionMask(SavedMask);
        {$endif}
    end;


finalization

    {$ifdef __EmbedDLL}

    { unload DLL first }
    PDFiumDone;

    if FullPdfiumDLLFileName <> '' then
    begin
        { Note: when debugging in IDE, DeleteFile will return True but the
          file only delete when the process terminates and then RemoveDir
          fails because it is not empty. Could not find workaround for this.
          When running outside the IDE, the file and folder are properly
          deleted }
        SysUtils.DeleteFile(FullPdfiumDLLFileName);

        SysUtils.RemoveDir( TempPDFiumFolder );
    end;
    {$endif}

end.


