{----------------------------------------------------------------------------
|
|
| Module: PdfiumAPI
|
| Description: Low level PDF access points standardize access to Pdfium library.
|              Created to allow using differnet PDFium libraries to keep
|              support for XP that uses different DLL with different
|              calling conventions
|
| History: Mar 24, 2022 Michel Brazeau
|
|---------------------------------------------------------------------------}
unit PdfiumAPI;

{$I Envision.Inc}


interface

uses
    {$ifdef __DelphiD13AndAbove}Winapi.Windows,{$else}Windows,{$endif}

    SysUtils; // for SIZE_T

const
    {$IFDEF CPUX64}
    CPdfiumXPDLLName = 'pdfium64xp.dll';
    CPdfiumDllName   = 'pdfium64.dll';
    {$else}
    CPdfiumXPDLLName = 'pdfium32xp.dll';
    CPdfiumDllName   = 'pdfium32.dll';
    {$endif}

  // More DIB formats
  FPDFBitmap_Gray = 1; // Gray scale bitmap, one byte per pixel.
  FPDFBitmap_BGR  = 2; // 3 bytes per pixel, byte order: blue, green, red.
  FPDFBitmap_BGRx = 3; // 4 bytes per pixel, byte order: blue, green, red, unused.
  FPDFBitmap_BGRA = 4; // 4 bytes per pixel, byte order: blue, green, red, alpha.
  // The page object constants.
  FPDF_PAGEOBJ_TEXT    = 1;
  FPDF_PAGEOBJ_PATH    = 2;
  FPDF_PAGEOBJ_IMAGE   = 3;
  FPDF_PAGEOBJ_SHADING = 4;
  FPDF_PAGEOBJ_FORM    = 5;
  FPDF_ERR_SUCCESS  = 0;    // No error.
  FPDF_ERR_UNKNOWN  = 1;    // Unknown error.
  FPDF_ERR_FILE     = 2;    // File not found or could not be opened.
  FPDF_ERR_FORMAT   = 3;    // File not in PDF format or corrupted.
  FPDF_ERR_PASSWORD = 4;    // Password required or incorrect password.
  FPDF_ERR_SECURITY = 5;    // Unsupported security scheme.
  FPDF_ERR_PAGE     = 6;    // Page not found or content error.
// Page rendering flags. They can be combined with bit OR.
  FPDF_ANNOT                    = $01;  // Set if annotations are to be rendered.
  FPDF_LCD_TEXT                 = $02;  // Set if using text rendering optimized for LCD display.
  FPDF_NO_NATIVETEXT            = $04;  // Don't use the native text output available on some platforms
  FPDF_GRAYSCALE                = $08;  // Grayscale output.
  FPDF_DEBUG_INFO               = $80;  // Set if you want to get some debug info.
  // Please discuss with Foxit first if you need to collect debug info.
  FPDF_NO_CATCH                 = $100; // Set if you don't want to catch exception.
  FPDF_RENDER_LIMITEDIMAGECACHE = $200; // Limit image cache size.
  FPDF_RENDER_FORCEHALFTONE     = $400; // Always use halftone for image stretching.
  FPDF_PRINTING                 = $800; // Render for printing.
  FPDF_REVERSE_BYTE_ORDER       = $10;  //set whether render in a reverse Byte order, this flag only
  //@brief Incremental.
  FPDF_INCREMENTAL     = 1;
  //@brief No Incremental.
  FPDF_NO_INCREMENTAL  = 2;
  //@brief Remove security.
  FPDF_REMOVE_SECURITY = 3;
  FLATTEN_FAIL       = 0;  // Flatten operation failed.
  FLATTEN_SUCCESS    = 1;  // Flatten operation succeed.
  FLATTEN_NOTINGTODO = 2;  // There is nothing can be flatten.
  // Flags used by FPDFText_FindStart function.
  FPDF_MATCHCASE      = $00000001;    //If not set, it will not match case by default.
  FPDF_MATCHWHOLEWORD = $00000002;    //If not set, it will not match the whole word by default.

  //flags for page mode.
  //Unknown value
  PAGEMODE_UNKNOWN        = -1;
  //Neither document outline nor thumbnail images visible
  PAGEMODE_USENONE        = 0;
  //Document outline visible
  PAGEMODE_USEOUTLINES    = 1;
  //Thumbnial images visible
  PAGEMODE_USETHUMBS      = 2;
  //Full-screen mode, with no menu bar, window controls, or any other window visible
  PAGEMODE_FULLSCREEN     = 3;
  //Optional content group panel visible
  PAGEMODE_USEOC          = 4;
  //Attachments panel visible
  PAGEMODE_USEATTACHMENTS = 5;

// The result of the process which check linearized PDF.
const
  FSDK_IS_LINEARIZED     = 1;
  FSDK_NOT_LINEARIZED    = 0;
  FSDK_UNKNOW_LINEARIZED = -1;

type
  // Data types
  __FPDF_PTRREC = record end;
  FPDF_MODULEMGR    = ^__FPDF_PTRREC;
  // PDF types
  FPDF_DOCUMENT     = ^__FPDF_PTRREC;
  FPDF_PAGE         = ^__FPDF_PTRREC;
  FPDF_PAGEOBJECT   = ^__FPDF_PTRREC; // Page object(text, path, etc)
  FPDF_PATH         = ^__FPDF_PTRREC;
  FPDF_CLIPPATH     = ^__FPDF_PTRREC;
  FPDF_BITMAP       = ^__FPDF_PTRREC;
  FPDF_FONT         = ^__FPDF_PTRREC;
  FPDF_TEXTPAGE     = ^__FPDF_PTRREC;
  FPDF_SCHHANDLE    = ^__FPDF_PTRREC;
  FPDF_PAGELINK     = ^__FPDF_PTRREC;
  FPDF_HMODULE      = ^__FPDF_PTRREC;
  FPDF_DOCSCHHANDLE = ^__FPDF_PTRREC;
  FPDF_BOOKMARK     = ^__FPDF_PTRREC;
  FPDF_DEST         = ^__FPDF_PTRREC;
  FPDF_ACTION       = ^__FPDF_PTRREC;
  FPDF_LINK         = ^__FPDF_PTRREC;
  FPDF_PAGERANGE    = ^__FPDF_PTRREC;
  // Basic data types
  FPDF_BOOL  = Integer;
  FPDF_ERROR = Integer;
  FPDF_DWORD = LongWord;
  FS_FLOAT = Single;
  // Duplex types
  FPDF_DUPLEXTYPE = (
    DuplexUndefined = 0,
    Simplex,
    DuplexFlipShortEdge,
    DuplexFlipLongEdge
  );
  // String types
  FPDF_WCHAR = WideChar;
  FPDF_LPCBYTE = PAnsiChar;
  // FPDFSDK may use three types of strings: byte string, wide string (UTF-16LE encoded), and platform dependent string
  FPDF_BYTESTRING = PAnsiChar;
  FPDF_WIDESTRING = PWideChar; // Foxit PDF SDK always use UTF-16LE encoding wide string,
                               // each character use 2 bytes (except surrogation), with low byte first.
  // For Windows programmers: for most case it's OK to treat FPDF_WIDESTRING as Windows unicode string,
  //     however, special care needs to be taken if you expect to process Unicode larger than 0xffff.
  // For Linux/Unix programmers: most compiler/library environment uses 4 bytes for a Unicode character,
  //    you have to convert between FPDF_WIDESTRING and system wide string by yourself.
  FPDF_STRING = PAnsiChar;
  PFPDF_LINK = ^FPDF_LINK; // array
  PFPDF_PAGE = ^FPDF_PAGE; // array
  // @brief Matrix for transformation.
  FS_MATRIX = record
    a: Single; //< @brief Coefficient a.
    b: Single; //< @brief Coefficient b.
    c: Single; //< @brief Coefficient c.
    d: Single; //< @brief Coefficient d.
    e: Single; //< @brief Coefficient e.
    f: Single; //< @brief Coefficient f.
  end;
  PFSMatrix = ^TFSMatrix;
  TFSMatrix = FS_MATRIX;
  // @brief Rectangle area(float) in device or page coordination system.
  PFS_RECTF = ^FS_RECTF;
  FS_RECTF = record
    //@brief The x-coordinate of the left-top corner.
    left: Single;
    //@brief The y-coordinate of the left-top corner.
    top: Single;
    //@brief The x-coordinate of the right-bottom corner.
    right: Single;
    //@brief The y-coordinate of the right-bottom corner.
    bottom: Single;
  end;
  // @brief Const Pointer to ::FS_RECTF structure.
  FS_LPCRECTF = ^FS_RECTF;
  PFSRectF = ^TFSRectF;
  TFSRectF = FS_RECTF;
  {$IF not declared(SIZE_T)}
  SIZE_T = LongWord;
  {$IFEND}
  {$IF not declared(DWORD)}
  DWORD = UInt16;
  {$IFEND}

  // Structure for custom file write
  PFPDF_FILEWRITE = ^FPDF_FILEWRITE;
  FPDF_FILEWRITE = record
    //Version number of the interface. Currently must be 1.
    //
    version: Integer;
    //
    // Method: WriteBlock
    //      Output a block of data in your custom way.
    // Interface Version:
    //      1
    // Implementation Required:
    //      Yes
    // Comments:
    //      Called by function FPDF_SaveDocument
    // Parameters:
    //      pThis    -  Pointer to the structure itself
    //      pData    -  Pointer to a buffer to output
    //      size     -  The size of the buffer.
    // Return value:
    //      Should be non-zero if successful, zero for error.
    //
    WriteBlock: function(pThis: PFPDF_FILEWRITE; pData: Pointer; size: LongWord): Integer; cdecl;
  end;
  PFPDFFileWrite = ^TFPDFFileWrite;
  TFPDFFileWrite = FPDF_FILEWRITE;
// Structure for custom file access.
type
  PFPDF_FILEACCESS = ^FPDF_FILEACCESS;
  FPDF_FILEACCESS = record
    // File length, in bytes.
    m_FileLen: LongWord;
    // A function pointer for getting a block of data from specific position.
    // Position is specified by byte offset from beginning of the file.
    // The position and size will never go out range of file length.
    // It may be possible for FPDFSDK to call this function multiple times for same position.
    // Return value: should be non-zero if successful, zero for error.
    m_GetBlock: function(param: Pointer; position: LongWord; pBuf: PByte; size: LongWord): Integer; cdecl;
    // A custom pointer for all implementation specific data.
    // This pointer will be used as the first parameter to m_GetBlock callback.
    m_Param: Pointer;
  end;
  PFPDFFileAccess = ^TFPDFFileAccess;
  TFPDFFileAccess = FPDF_FILEACCESS;
const
  //flags for type of unsupport object.
  FPDF_UNSP_DOC_XFAFORM               = 1;
  FPDF_UNSP_DOC_PORTABLECOLLECTION    = 2;
  FPDF_UNSP_DOC_ATTACHMENT            = 3;
  FPDF_UNSP_DOC_SECURITY              = 4;
  FPDF_UNSP_DOC_SHAREDREVIEW          = 5;
  FPDF_UNSP_DOC_SHAREDFORM_ACROBAT    = 6;
  FPDF_UNSP_DOC_SHAREDFORM_FILESYSTEM = 7;
  FPDF_UNSP_DOC_SHAREDFORM_EMAIL      = 8;
  FPDF_UNSP_ANNOT_3DANNOT             = 11;
  FPDF_UNSP_ANNOT_MOVIE               = 12;
  FPDF_UNSP_ANNOT_SOUND               = 13;
  FPDF_UNSP_ANNOT_SCREEN_MEDIA        = 14;
  FPDF_UNSP_ANNOT_SCREEN_RICHMEDIA    = 15;
  FPDF_UNSP_ANNOT_ATTACHMENT          = 16;
  FPDF_UNSP_ANNOT_SIG                 = 17;
type
  PUNSUPPORT_INFO = ^UNSUPPORT_INFO;
  UNSUPPORT_INFO = record
    //**
    //* Version number of the interface. Currently must be 1.
    //**
    version: Integer;
    //**
    //* Method: FSDK_UnSupport_Handler
    //*       UnSupport Object process handling function.
    //* Interface Version:
    //*      1
    //* Implementation Required:
    //*      Yes
    //* Parameters:
    //*    pThis    -  Pointer to the interface structure itself.
    //*    nType    -  The type of unsupportObject
    //*   Return value:
    //*     None.
    //**
    FSDK_UnSupport_Handler: procedure(pThis: PUNSUPPORT_INFO; nType: Integer); cdecl;
  end;
  PUnsupportInfo = ^TUnsupportInfo;
  TUnsupportInfo = UNSUPPORT_INFO;


type
  TImportFuncRec = record
    P: PPointer;
    N: PAnsiChar;
  end;
type
  {$IF not declared(SIZE_T)}
  SIZE_T = LongWord;
  {$IFEND}
  {$IF not declared(DWORD)}
  DWORD = UInt16;
  {$IFEND}

  //**
  //* Interface: FX_FILEAVAIL
  //*      Interface for checking whether the section of the file is available.
  //**
  PFX_FILEAVAIL = ^FX_FILEAVAIL;
  FX_FILEAVAIL = record
    //**
    //* Version number of the interface. Currently must be 1.
    //**
    version: Integer;
    //**
    //* Method: IsDataAvail
    //*    Report whether the specified data section is available. A section is available only if all bytes in the section is available.
    //* Interface Version:
    //*    1
    //* Implementation Required:
    //*    Yes
    //* Parameters:
    //*    pThis     -  Pointer to the interface structure itself.
    //*    offset    -  The offset of the data section in the file.
    //*    size      -  The size of the data section
    //* Return Value:
    //*    true means the specified data section is available.
    //* Comments:
    //*    Called by Foxit SDK to check whether the data section is ready.
    //**
    IsDataAvail: function(pThis: PFX_FILEAVAIL; offset, size: SIZE_T): ByteBool; cdecl;
  end;
  PFXFileAvail = ^TFXFileAvail;
  TFXFileAvail = FX_FILEAVAIL;
  FPDF_AVAIL = ^__FPDF_PTRREC;
//**
//* Interface: FX_DOWNLOADHINTS
//*      Download hints interface. Used to receive hints for further downloading.
//**
type
  PFX_DOWNLOADHINTS = ^FX_DOWNLOADHINTS;
  FX_DOWNLOADHINTS = record
    //**
    //* Version number of the interface. Currently must be 1.
    //**
    version: Integer;
    //**
    //* Method: AddSegment
    //*    Add a section to be downloaded.
    //* Interface Version:
    //*    1
    //* Implementation Required:
    //*    Yes
    //* Parameters:
    //*    pThis    -  Pointer to the interface structure itself.
    //*    offset   -  The offset of the hint reported to be downloaded.
    //*    size     -  The size of the hint reported to be downloaded.
    //* Return Value:
    //*    None.
    //* Comments:
    //*    Called by Foxit SDK to report some downloading hints for download manager.
    //*    The position and size of section may be not accurate, part of the section might be already available.
    //*    The download manager must deal with that to maximize download efficiency.
    //**
    AddSegment: procedure(pThis: PFX_DOWNLOADHINTS; offset, size: SIZE_T); cdecl;
  end;
  PFXDownloadHints = ^TFXDownloadHints;
  TFXDownloadHints = FX_DOWNLOADHINTS;

function FPDF_ARGB(a, r, g, b: Byte): DWORD;
function FPDF_GetBValue(argb: DWORD): Byte;
function FPDF_GetGValue(argb: DWORD): Byte;
function FPDF_GetRValue(argb: DWORD): Byte;
function FPDF_GetAValue(argb: DWORD): Byte;

resourcestring
  RsFailedToLoadProc = 'Symbol "%s" was not found in pdfium.dll';
  RsPdfiumNotLoaded = 'pdfium.dll is not loaded';

procedure InitPDFium(const DllPath: string = '');
procedure PdfiumDone;

function PdfSetUnSpObjProcessHandler(unsp_info: PUNSUPPORT_INFO): FPDF_BOOL;

function PdfGetLastError : LongWord;

procedure PdfCloseDocument(document: FPDF_DOCUMENT);

function PdfLoadCustomDocument(pFileAccess: PFPDF_FILEACCESS; password: FPDF_BYTESTRING): FPDF_DOCUMENT;

function PdfGetPageCount(document: FPDF_DOCUMENT): Integer;

function PdfLoadMemDocument(data_buf: Pointer; size: Integer; password: FPDF_BYTESTRING): FPDF_DOCUMENT;

function PdfLoadPage(document: FPDF_DOCUMENT; page_index: Integer): FPDF_PAGE;

function PdfSaveWithVersion(document: FPDF_DOCUMENT; pFileWrite: PFPDF_FILEWRITE;
    flags: FPDF_DWORD; fileVersion: Integer): FPDF_BOOL;

function PdfSaveAsCopy(document: FPDF_DOCUMENT; pFileWrite: PFPDF_FILEWRITE; flags: FPDF_DWORD): FPDF_BOOL;

function PdfCreateNewDocument : FPDF_DOCUMENT;

procedure PdfPage_Delete(document: FPDF_DOCUMENT; page_index: Integer);

function PdfPage_New(document: FPDF_DOCUMENT; page_index: Integer; width, height: Double): FPDF_PAGE;
procedure PdfClosePage(page: FPDF_PAGE);

function PdfGetMetaText(doc: FPDF_DOCUMENT; tag: FPDF_BYTESTRING; buffer: Pointer; buflen: LongWord): LongWord;

function PdfGetFileVersion(doc: FPDF_DOCUMENT; var fileVersion: Integer): FPDF_BOOL;

function PdfGetPageSizeByIndex(document: FPDF_DOCUMENT; page_index: Integer; var width, height: Double): Integer;

procedure PdfText_FindClose(handle: FPDF_SCHHANDLE);
function PdfLink_LoadWebLinks(text_page: FPDF_TEXTPAGE): FPDF_PAGELINK;
function PdfLink_CountWebLinks(link_page: FPDF_PAGELINK): Integer;
procedure PdfLink_CloseWebLinks(link_page: FPDF_PAGELINK);
function PdfLink_GetURL(link_page: FPDF_PAGELINK; link_index: Integer; buffer: PWideChar; buflen: Integer): Integer;
procedure PdfText_ClosePage(text_page: FPDF_TEXTPAGE);

procedure PdfRenderPage(DC: HDC; page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; flags: Integer);

procedure PdfRenderPageBitmap(bitmap: FPDF_BITMAP; page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; flags: Integer);

function PdfGetPageWidth(page: FPDF_PAGE): Double;

function PdfGetPageHeight(page: FPDF_PAGE): Double;

function PdfPage_HasTransparency(page: FPDF_PAGE): FPDF_BOOL;

procedure PdfDeviceToPage(page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; device_x, device_y: Integer; var page_x, page_y: Double);
procedure PdfPageToDevice(page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; page_x, page_y: Double; var device_x, device_y: Integer);

function PdfPage_GetRotation(page: FPDF_PAGE): Integer;
procedure  PdfPage_SetRotation(page: FPDF_PAGE; rotate: Integer);

function PdfPage_GenerateContent(page: FPDF_PAGE): FPDF_BOOL;
function PdfText_LoadPage(page: FPDF_PAGE): FPDF_TEXTPAGE;


function PdfText_FindStart(text_page: FPDF_TEXTPAGE; findwhat: FPDF_WIDESTRING; flags: LongWord;
    start_index: Integer): FPDF_SCHHANDLE;
function PdfText_FindNext(handle: FPDF_SCHHANDLE): FPDF_BOOL;
function PdfText_FindPrev(handle: FPDF_SCHHANDLE): FPDF_BOOL;

function PdfText_GetSchResultIndex(handle: FPDF_SCHHANDLE): Integer;

function PdfText_GetSchCount(handle: FPDF_SCHHANDLE): Integer;
function PdfLink_CountRects(link_page: FPDF_PAGELINK; link_index: Integer): Integer;
procedure PdfLink_GetRect(link_page: FPDF_PAGELINK; link_index, rect_index: Integer;
    var left, top, right, bottom: Double);
function PdfText_GetCharIndexFromTextIndex(text_page: FPDF_TEXTPAGE; nTextIndex: Integer): Integer;
function PdfText_CountChars(text_page: FPDF_TEXTPAGE): Integer;

function PdfText_GetUnicode(text_page: FPDF_TEXTPAGE; index: Integer): WideChar;
function PdfText_GetFontSize(text_page: FPDF_TEXTPAGE; index: Integer): Double;
procedure PdfText_GetCharBox(text_page: FPDF_TEXTPAGE; index: Integer; var left, right, bottom, top: Double);
function PdfText_GetCharIndexAtPos(text_page: FPDF_TEXTPAGE; x, y, xTorelance, yTolerance: Double): Integer;
function PdfText_GetText(text_page: FPDF_TEXTPAGE; start_index, count: Integer; OutResult: PWideChar): Integer;
function PdfText_CountRects(text_page: FPDF_TEXTPAGE; start_index, count: Integer): Integer;
procedure PdfText_GetRect(text_page: FPDF_TEXTPAGE; rect_index: Integer; var left, top, right, bottom: Double);
function PdfText_GetBoundedText(text_page: FPDF_TEXTPAGE; left, top, right, bottom: Double;
    buffer: PWideChar; buflen: Integer): Integer;

function PdfBitmap_GetWidth(bitmap: FPDF_BITMAP): Integer;
function PdfBitmap_GetHeight(bitmap: FPDF_BITMAP): Integer;
function PdfBitmap_GetStride(bitmap: FPDF_BITMAP): Integer;
procedure PdfBitmap_Destroy(bitmap: FPDF_BITMAP);
function PdfBitmap_Create(width, height: Integer; alpha: Integer): FPDF_BITMAP;
function PdfBitmap_CreateEx(width, height: Integer; format: Integer; first_scan: Pointer;
    stride: Integer): FPDF_BITMAP;
procedure PdfBitmap_FillRect(bitmap: FPDF_BITMAP; left, top, width, height: Integer; color: FPDF_DWORD);
function PdfBitmap_GetBuffer(bitmap: FPDF_BITMAP): Pointer;


implementation

uses
    EnMisc,      { for AddBackSlashToPath}
    PdfiumLib,
    PdfiumLibXP;

var
    DLLLoaded   : Boolean;
    UsingXPDLL  : Boolean;


const
    CInitPDFiumNotCalled = 'PdfiumAPI.InitPDFium has not been called';

function FPDF_ARGB(a, r, g, b: Byte): DWORD;
begin
  Result := DWORD(b) or (DWORD(g) shl 8) or (DWORD(r) shl 16) or (DWORD(a) shl 24);
end;
function FPDF_GetBValue(argb: DWORD): Byte;
begin
  Result := Byte(argb);
end;
function FPDF_GetGValue(argb: DWORD): Byte;
begin
  Result := Byte(argb shr 8);
end;
function FPDF_GetRValue(argb: DWORD): Byte;
begin
  Result := Byte(argb shr 16);
end;
function FPDF_GetAValue(argb: DWORD): Byte;
begin
  Result := Byte(argb shr 24);
end;


{--------------------------------------------------------------------------}

procedure InitPDFium(const DllPath: string = '');
var
    DLLFileName   : String;


begin
    if DLLLoaded then
        Exit;

    { check if XP DLL is present, if so, use XP version }
    DLLFileName := AddBackSlashToPath( ExtractFilePath(ParamStr(0)) ) + CPdfiumXPDLLName;
    if FileExists(DLLFileName) then
    begin
        PdfiumLibXP.InitPDFiumUnit;
        UsingXPDLL  := True;
    end
    else
    begin
        PdfiumLib.InitPDFiumUnit(DllPath);
        UsingXPDLL  := False;
    end;
    DLLLoaded   := True;
end;

{--------------------------------------------------------------------------}

procedure PdfiumDone;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.PdfiumUnitDone
    else
        PdfiumLib.PdfiumUnitDone;
end;


{--------------------------------------------------------------------------}

function PdfSetUnSpObjProcessHandler(unsp_info: PUNSUPPORT_INFO): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FSDK_SetUnSpObjProcessHandler( unsp_info )
    else
        Result := PdfiumLib.FSDK_SetUnSpObjProcessHandler( unsp_info );
end;

{--------------------------------------------------------------------------}

function PdfGetLastError : LongWord;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetLastError
    else
        Result := PdfiumLib.FPDF_GetLastError;
end;

{--------------------------------------------------------------------------}

procedure PdfCloseDocument(document: FPDF_DOCUMENT);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_CloseDocument(document)
    else
        PdfiumLib.FPDF_CloseDocument(document);
end;

{--------------------------------------------------------------------------}

function PdfLoadCustomDocument(pFileAccess: PFPDF_FILEACCESS; password: FPDF_BYTESTRING): FPDF_DOCUMENT;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_LoadCustomDocument( pFileAccess, password )
    else
        Result := PdfiumLib.FPDF_LoadCustomDocument( pFileAccess, password );
end;

{--------------------------------------------------------------------------}

function PdfGetPageCount(document: FPDF_DOCUMENT): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetPageCount( document )
    else
        Result := PdfiumLib.FPDF_GetPageCount( document );
end;

{--------------------------------------------------------------------------}

function PdfLoadMemDocument(data_buf: Pointer; size: Integer; password: FPDF_BYTESTRING): FPDF_DOCUMENT;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_LoadMemDocument( data_buf, size, password )
    else
        Result := PdfiumLib.FPDF_LoadMemDocument( data_buf, size, password );
end;

{--------------------------------------------------------------------------}

function PdfLoadPage(document: FPDF_DOCUMENT; page_index: Integer): FPDF_PAGE;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_LoadPage( document, page_index )
    else
        Result := PdfiumLib.FPDF_LoadPage( document, page_index );
end;

{--------------------------------------------------------------------------}

function PdfSaveWithVersion(document: FPDF_DOCUMENT; pFileWrite: PFPDF_FILEWRITE;
    flags: FPDF_DWORD; fileVersion: Integer): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_SaveWithVersion( document, pFileWrite, flags, fileVersion )
    else
        Result := PdfiumLib.FPDF_SaveWithVersion( document, pFileWrite, flags, fileVersion );
end;

{--------------------------------------------------------------------------}

function PdfSaveAsCopy(document: FPDF_DOCUMENT; pFileWrite: PFPDF_FILEWRITE; flags: FPDF_DWORD): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_SaveAsCopy( document, pFileWrite, flags )
    else
        Result := PdfiumLib.FPDF_SaveAsCopy( document, pFileWrite, flags );
end;

{--------------------------------------------------------------------------}

function PdfCreateNewDocument : FPDF_DOCUMENT;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_CreateNewDocument
    else
        Result := PdfiumLib.FPDF_CreateNewDocument;
end;

{--------------------------------------------------------------------------}

procedure PdfPage_Delete(document: FPDF_DOCUMENT; page_index: Integer);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFPage_Delete( document, page_index )
    else
        PdfiumLib.FPDFPage_Delete( document, page_index );
end;

{--------------------------------------------------------------------------}

function PdfPage_New(document: FPDF_DOCUMENT; page_index: Integer; width, height: Double): FPDF_PAGE;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFPage_New( document, page_index, width, height  )
    else
        Result := PdfiumLib.FPDFPage_New( document, page_index, width, height  );
end;

{--------------------------------------------------------------------------}

procedure PdfClosePage(page: FPDF_PAGE);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_ClosePage( page )
    else
        PdfiumLib.FPDF_ClosePage( page )
end;

{--------------------------------------------------------------------------}

function PdfGetMetaText(doc: FPDF_DOCUMENT; tag: FPDF_BYTESTRING; buffer: Pointer; buflen: LongWord): LongWord;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetMetaText( doc, tag, buffer, buflen  )
    else
        Result := PdfiumLib.FPDF_GetMetaText( doc, tag, buffer, buflen  );
end;

{--------------------------------------------------------------------------}

function PdfGetFileVersion(doc: FPDF_DOCUMENT; var fileVersion: Integer): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetFileVersion( doc, fileVersion )
    else
        Result := PdfiumLib.FPDF_GetFileVersion( doc, fileVersion );
end;

{--------------------------------------------------------------------------}

function PdfGetPageSizeByIndex(document: FPDF_DOCUMENT; page_index: Integer; var width, height: Double): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetPageSizeByIndex( document, page_index, width, height )
    else
        Result := PdfiumLib.FPDF_GetPageSizeByIndex( document, page_index, width, height );
end;

{--------------------------------------------------------------------------}

procedure PdfText_FindClose(handle: FPDF_SCHHANDLE);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFText_FindClose( handle )
    else
        PdfiumLib.FPDFText_FindClose( handle )
end;

{--------------------------------------------------------------------------}

function PdfLink_LoadWebLinks(text_page: FPDF_TEXTPAGE): FPDF_PAGELINK;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFLink_LoadWebLinks( text_page )
    else
        Result := PdfiumLib.FPDFLink_LoadWebLinks( text_page );
end;

{--------------------------------------------------------------------------}

function PdfLink_CountWebLinks(link_page: FPDF_PAGELINK): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFLink_CountWebLinks( link_page )
    else
        Result := PdfiumLib.FPDFLink_CountWebLinks( link_page );
end;
{--------------------------------------------------------------------------}


procedure PdfLink_CloseWebLinks(link_page: FPDF_PAGELINK);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFLink_CloseWebLinks( link_page )
    else
        PdfiumLib.FPDFLink_CloseWebLinks( link_page )
end;

{--------------------------------------------------------------------------}

function PdfLink_GetURL(link_page: FPDF_PAGELINK; link_index: Integer; buffer: PWideChar; buflen: Integer): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFLink_CountWebLinks( link_page )
    else
        Result := PdfiumLib.FPDFLink_CountWebLinks( link_page );
end;

{--------------------------------------------------------------------------}

procedure PdfText_ClosePage(text_page: FPDF_TEXTPAGE);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFText_ClosePage( text_page )
    else
        PdfiumLib.FPDFText_ClosePage( text_page )
end;

{--------------------------------------------------------------------------}

procedure PdfRenderPage(DC: HDC; page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; flags: Integer);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_RenderPage( DC, page, start_x, start_y, size_x, size_y, rotate, flags )
    else
        PdfiumLib.FPDF_RenderPage( DC, page, start_x, start_y, size_x, size_y, rotate, flags );
end;

{--------------------------------------------------------------------------}

procedure PdfRenderPageBitmap(bitmap: FPDF_BITMAP; page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; flags: Integer);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_RenderPageBitmap( bitmap, page, start_x, start_y, size_x, size_y, rotate, flags )
    else
        PdfiumLib.FPDF_RenderPageBitmap( bitmap, page, start_x, start_y, size_x, size_y, rotate, flags );
end;

{--------------------------------------------------------------------------}

function PdfGetPageWidth(page: FPDF_PAGE): Double;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetPageWidth( page )
    else
        Result := PdfiumLib.FPDF_GetPageWidth( page );
end;

{--------------------------------------------------------------------------}

function PdfGetPageHeight(page: FPDF_PAGE): Double;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDF_GetPageHeight( page )
    else
        Result := PdfiumLib.FPDF_GetPageHeight( page );
end;

{--------------------------------------------------------------------------}

function PdfPage_HasTransparency(page: FPDF_PAGE): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFPage_HasTransparency( page )
    else
        Result := PdfiumLib.FPDFPage_HasTransparency( page );
end;

{--------------------------------------------------------------------------}

procedure PdfDeviceToPage(page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; device_x, device_y: Integer; var page_x, page_y: Double);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_DeviceToPage( page, start_x, start_y, size_x, size_y, rotate, device_x, device_y, page_x, page_y )
    else
        PdfiumLib.FPDF_DeviceToPage( page, start_x, start_y, size_x, size_y, rotate, device_x, device_y, page_x, page_y );
end;

{--------------------------------------------------------------------------}

procedure PdfPageToDevice(page: FPDF_PAGE; start_x, start_y, size_x, size_y: Integer;
    rotate: Integer; page_x, page_y: Double; var device_x, device_y: Integer);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDF_PageToDevice( page, start_x, start_y, size_x, size_y, rotate, page_x, page_y, device_x, device_y )
    else
        PdfiumLib.FPDF_PageToDevice( page, start_x, start_y, size_x, size_y, rotate, page_x, page_y, device_x, device_y );
end;

{--------------------------------------------------------------------------}

function PdfPage_GetRotation(page: FPDF_PAGE): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFPage_GetRotation( page )
    else
        Result := PdfiumLib.FPDFPage_GetRotation( page );
end;

{--------------------------------------------------------------------------}

procedure  PdfPage_SetRotation(page: FPDF_PAGE; rotate: Integer);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFPage_SetRotation( page, rotate )
    else
        PdfiumLib.FPDFPage_SetRotation( page, rotate );
end;

{--------------------------------------------------------------------------}

function PdfPage_GenerateContent(page: FPDF_PAGE): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFPage_GenerateContent( page )
    else
        Result := PdfiumLib.FPDFPage_GenerateContent( page );
end;

{--------------------------------------------------------------------------}

function PdfText_LoadPage(page: FPDF_PAGE): FPDF_TEXTPAGE;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_LoadPage( page )
    else
        Result := PdfiumLib.FPDFText_LoadPage( page );
end;

{--------------------------------------------------------------------------}

function PdfText_FindStart(text_page: FPDF_TEXTPAGE; findwhat: FPDF_WIDESTRING; flags: LongWord;
    start_index: Integer): FPDF_SCHHANDLE;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_FindStart( text_page, findwhat, flags, start_index )
    else
        Result := PdfiumLib.FPDFText_FindStart( text_page, findwhat, flags, start_index );
end;

{--------------------------------------------------------------------------}

function PdfText_FindNext(handle: FPDF_SCHHANDLE): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_FindNext( handle )
    else
        Result := PdfiumLib.FPDFText_FindNext( handle );
end;

{--------------------------------------------------------------------------}

function PdfText_FindPrev(handle: FPDF_SCHHANDLE): FPDF_BOOL;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_FindPrev( handle )
    else
        Result := PdfiumLib.FPDFText_FindPrev( handle );
end;

{--------------------------------------------------------------------------}

function PdfText_GetSchResultIndex(handle: FPDF_SCHHANDLE): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetSchResultIndex( handle )
    else
        Result := PdfiumLib.FPDFText_GetSchResultIndex( handle );
end;

{--------------------------------------------------------------------------}

function PdfText_GetSchCount(handle: FPDF_SCHHANDLE): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetSchCount( handle )
    else
        Result := PdfiumLib.FPDFText_GetSchCount( handle );
end;

{--------------------------------------------------------------------------}

function PdfLink_CountRects(link_page: FPDF_PAGELINK; link_index: Integer): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFLink_CountRects( link_page, link_index )
    else
        Result := PdfiumLib.FPDFLink_CountRects( link_page, link_index );
end;

{--------------------------------------------------------------------------}

procedure PdfLink_GetRect( link_page: FPDF_PAGELINK; link_index, rect_index: Integer;
    var left, top, right, bottom: Double);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        PdfiumLibXP.FPDFLink_GetRect( link_page, link_index, rect_index, left, top, right, bottom )
    else
        PdfiumLib.FPDFLink_GetRect( link_page, link_index, rect_index, left, top, right, bottom );

end;

{--------------------------------------------------------------------------}

function PdfText_GetCharIndexFromTextIndex(text_page: FPDF_TEXTPAGE; nTextIndex: Integer): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );

    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetCharIndexFromTextIndex( text_page, nTextIndex )
    else
        Result := PdfiumLib.FPDFText_GetCharIndexFromTextIndex( text_page, nTextIndex );
end;

{--------------------------------------------------------------------------}

function PdfText_CountChars(text_page: FPDF_TEXTPAGE): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_CountChars( text_page )
    else
        Result := PdfiumLib.FPDFText_CountChars( text_page );
end;
{--------------------------------------------------------------------------}

function PdfText_GetUnicode(text_page: FPDF_TEXTPAGE; index: Integer): WideChar;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetUnicode( text_page, index )
    else
        Result := PdfiumLib.FPDFText_GetUnicode( text_page, index );
end;
{--------------------------------------------------------------------------}

function PdfText_GetFontSize(text_page: FPDF_TEXTPAGE; index: Integer): Double;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetFontSize( text_page, index )
    else
        Result := PdfiumLib.FPDFText_GetFontSize( text_page, index );
end;
{--------------------------------------------------------------------------}

procedure PdfText_GetCharBox(text_page: FPDF_TEXTPAGE; index: Integer; var left, right, bottom, top: Double);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        PdfiumLibXP.FPDFText_GetCharBox( text_page, index, left, right, bottom, top )
    else
        PdfiumLib.FPDFText_GetCharBox( text_page, index, left, right, bottom, top );
end;
{--------------------------------------------------------------------------}

function PdfText_GetCharIndexAtPos(text_page: FPDF_TEXTPAGE; x, y, xTorelance, yTolerance: Double): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetCharIndexAtPos( text_page, x, y, xTorelance, yTolerance )
    else
        Result := PdfiumLib.FPDFText_GetCharIndexAtPos( text_page, x, y, xTorelance, yTolerance );
end;
{--------------------------------------------------------------------------}

function PdfText_GetText(text_page: FPDF_TEXTPAGE; start_index, count: Integer; OutResult: PWideChar): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetText( text_page, start_index, count, OutResult )
    else
        Result := PdfiumLib.FPDFText_GetText( text_page, start_index, count, OutResult );
end;
{--------------------------------------------------------------------------}

function PdfText_CountRects(text_page: FPDF_TEXTPAGE; start_index, count: Integer): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_CountRects( text_page, start_index, count )
    else
        Result := PdfiumLib.FPDFText_CountRects( text_page, start_index, count );
end;
{--------------------------------------------------------------------------}

procedure PdfText_GetRect(text_page: FPDF_TEXTPAGE; rect_index: Integer; var left, top, right, bottom: Double);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        PdfiumLibXP.FPDFText_GetRect( text_page, rect_index, left, top, right, bottom )
    else
        PdfiumLib.FPDFText_GetRect( text_page, rect_index, left, top, right, bottom );
end;
{--------------------------------------------------------------------------}

function PdfText_GetBoundedText(text_page: FPDF_TEXTPAGE; left, top, right, bottom: Double;
    buffer: PWideChar; buflen: Integer): Integer;

begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFText_GetBoundedText( text_page, left, top, right, bottom, buffer, buflen )
    else
        Result := PdfiumLib.FPDFText_GetBoundedText( text_page, left, top, right, bottom, buffer, buflen );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_GetWidth(bitmap: FPDF_BITMAP): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_GetWidth( bitmap )
    else
        Result := PdfiumLib.FPDFBitmap_GetWidth( bitmap );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_GetHeight(bitmap: FPDF_BITMAP): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_GetHeight( bitmap )
    else
        Result := PdfiumLib.FPDFBitmap_GetHeight( bitmap );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_GetStride(bitmap: FPDF_BITMAP): Integer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_GetStride( bitmap )
    else
        Result := PdfiumLib.FPDFBitmap_GetStride( bitmap );
end;
{--------------------------------------------------------------------------}

procedure PdfBitmap_Destroy(bitmap: FPDF_BITMAP);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        PdfiumLibXP.FPDFBitmap_Destroy( bitmap )
    else
        PdfiumLib.FPDFBitmap_Destroy( bitmap );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_Create(width, height: Integer; alpha: Integer): FPDF_BITMAP;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_Create( width, height, alpha )
    else
        Result := PdfiumLib.FPDFBitmap_Create( width, height, alpha );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_CreateEx(width, height: Integer; format: Integer; first_scan: Pointer;
    stride: Integer): FPDF_BITMAP;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_CreateEx( width, height, format, first_scan, stride )
    else
        Result := PdfiumLib.FPDFBitmap_CreateEx( width, height, format, first_scan, stride );
end;
{--------------------------------------------------------------------------}

procedure PdfBitmap_FillRect(bitmap: FPDF_BITMAP; left, top, width, height: Integer; color: FPDF_DWORD);
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        PdfiumLibXP.FPDFBitmap_FillRect( bitmap, left, top, width, height, color )
    else
        PdfiumLib.FPDFBitmap_FillRect( bitmap, left, top, width, height, color );
end;
{--------------------------------------------------------------------------}

function PdfBitmap_GetBuffer(bitmap: FPDF_BITMAP): Pointer;
begin
    RaiseIfNot( DLLLoaded, CInitPdfiumNotCalled );
    if UsingXPDLL then
        Result := PdfiumLibXP.FPDFBitmap_GetBuffer( bitmap )
    else
        Result := PdfiumLib.FPDFBitmap_GetBuffer( bitmap );
end;
{--------------------------------------------------------------------------}


initialization
    DLLLoaded   := False;
    UsingXPDLL  := False;


end.
